/*
 * Copyright (C) 2011 Hewlett-Packard Development Company, L.P.
 *
 * This file is part of hpsa_obdr_mode.
 *
 * hpsa_obdr_mode is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * hpsa_obdr_mode is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with hpsa_obdr_mode; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * Author: Stephen M. Cameron <scameron@beardog.cce.hp.com>
 *
 */

/* Some versions of the hpsa driver do not properly
 * detect OBDR devices.
 *
 * Thus, when in OBDR mode, when the disaster recovery
 * software tries to change the tape drive out of OBDR
 * mode, it is unable to talk to it because the driver
 * didn't find it.
 *
 * This program hunts down hpsa controlled smart arrays,
 * then for each smart array, does a report physical luns.
 * Then for each cd-rom/tape drive, it does
 * an inquiry to determine if it's an OBDR device.
 *
 * Then, it sends a mode select to take it out of or 
 * put it in of OBDR mode.
 *
 * So running this program with parameters -m tape /dev/sg*
 * will find all the OBDR devices that the buggy hpsa
 * driver missed, and flip them out of or into OBDR mode.
 *
 * Then, via echo 1 > /sys/bus/scsi/....hostn/rescan,
 * the tape drives can be brought on-line.
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <string.h>
#include <stdint.h>
#include <arpa/inet.h> /* for htonl, etc. */
#include <errno.h>
#include <getopt.h>
#include <scsi/sg.h>
#include <scsi/scsi_ioctl.h>

/* Some old versions of kernel headers don't arrange to have __user,
 * which appears in cciss_ioctl.h, handled correctly by userland
 * programs.  So we handle it here.
 */
#define __user
#include <linux/cciss_ioctl.h>

static int debug = 0;
static char *progname = NULL;

#define QUERY 0
#define TAPEMODE 1
#define CDMODE 2

#ifndef VERSION
#warning VERSION not defined, using 'unknown'
#define VERSION "unknown"
#endif

static unsigned char *CTLR_LUNID = (unsigned char *) "\0\0\0\0\0\0\0\0";

#define HPSA_MAX_LUN 1024
#define REPORT_PHYSICAL 0xc3

#pragma pack(1)
struct ReportLUNdata {
	uint8_t LUNListLength[4];
	uint8_t reserved[4];
	uint8_t LUN[8 * HPSA_MAX_LUN];
};
#pragma pack()

static void setup_sgio(sg_io_hdr_t *sgio,
		unsigned char *cdb, unsigned char cdblen,
		unsigned char *databuffer, unsigned int databufferlen,
		unsigned char *sensebuffer, unsigned char sensebufferlen,
		int direction
	       	)
{
	memset(sgio, 0, sizeof(*sgio));
	sgio->interface_id = 'S';
	sgio->dxfer_direction = direction;
	sgio->cmd_len = cdblen;
	sgio->mx_sb_len = sensebufferlen;
	sgio->iovec_count = 0;
	sgio->dxfer_len = databufferlen;
	sgio->dxferp = databuffer;
	sgio->cmdp = cdb;
	sgio->sbp = sensebuffer;
	/* sgio->timeout = 0xffffffff; // no timeout */
	sgio->timeout = 0x0fffffff; /* At least one mptlinux controller doesn't like */
				    /* timeout of 0xffffffff, 1st drive, ok, 2nd, 3rd, etc */
				    /* sort of pretend to timeout instantly and abort cmds */
				    /* but oddly, return data, for inquiries anyhow. */ 
				    /* timeout of 0x0fffffff seems to make it happy. */
	sgio->flags = 0;
	sgio->pack_id = 0;
	sgio->usr_ptr = NULL;
	sgio->status = 0;
	sgio->masked_status = 0;
	sgio->msg_status = 0;
	sgio->sb_len_wr = 0;
	sgio->host_status = 0;
	sgio->driver_status = 0;
	sgio->resid = 0;
	sgio->duration = 0;
	sgio->info = 0;
}

static int debug_sgio()
{
	return 0;
}

static int do_sg_io(int fd, unsigned char *cdb, unsigned char cdblen,
	unsigned char *buffer, unsigned char buf_size, int direction)
{
	int status;
	sg_io_hdr_t sgio;
	unsigned char sensebuffer[64];

	memset(buffer, 0, buf_size);
	memset(&sgio, 0, sizeof(sgio));

	setup_sgio(&sgio, (unsigned char *) cdb, cdblen, buffer,
		buf_size, sensebuffer, sizeof(sensebuffer), direction);

	status = ioctl(fd, SG_IO, &sgio);

	if (status == 0 && sgio.host_status == 0 && sgio.driver_status == 0) {   /* cmd succeeded */
		if (sgio.status == 0 || (sgio.status == 2 &&
			(((sensebuffer[2] & 0x0f) == 0x00) || /* no error */
			 ((sensebuffer[2] & 0x0f) == 0x01)))) {
			return 0;
		}
		if (debug_sgio())
			fprintf(stderr, "sgio cmd 0x%02x check condition, sense key = %d\n",
				cdb[0], sensebuffer[2] & 0x0f);
		return -1;
	}
	if (debug_sgio())
		fprintf(stderr, "sgio ioctl: %d, cdb[0]=0x%02x, "
			"status host/driver/scsi/sensekey = %d/%d/%d/0x%02x\n",
			status, cdb[0], sgio.host_status, sgio.driver_status,
			sgio.status, sensebuffer[2]);
	return -1;
}

static int sgio_do_inquiry(int fd,
               unsigned char inquiry_page,
               unsigned char* inquiry_buf,
               unsigned char buf_size)
{
	unsigned char cdb[6];

	memset(inquiry_buf, 0, buf_size);
	memset(cdb, 0, sizeof(cdb));
	cdb[0] = 0x12;
	if (inquiry_page != 0) /* EVPD? */
		cdb[1] = 1;
	else
		cdb[1] = 0;
	cdb[2] = inquiry_page;
	cdb[4] = buf_size;

	return do_sg_io(fd, cdb, 6, inquiry_buf, buf_size, SG_DXFER_FROM_DEV);
}

static int is_smartarray(int fd)
{
	int rc;
	struct _cciss_pci_info_struct pciinfo;
	unsigned char inqbuf[36];
	
	/* test if it's a smart array (that driver supports this ioctl.) */
	rc = ioctl(fd, CCISS_GETPCIINFO, &pciinfo);
	if (rc) {
		if (debug)
			fprintf(stderr, "Not a smart array\n");
		return 0;
	}

	/* test if it's a smart array *controller* (and not logical drive,
	 * tape drive, etc.) 
	 */
	
	rc = sgio_do_inquiry(fd, 0, inqbuf, sizeof(inqbuf));
	if (rc) {
		if (debug)
			fprintf(stderr, "SG_IO inquiry failed.\n");
		return 0;
	}

	if (debug)
		fprintf(stderr, "Device is %sa RAID controller device\n",
			((inqbuf[0] & 0x1f) == 0x0C) ? "" : "not ");
	/* See if it's a RAID controller (device type 0x0C) */
	return ((inqbuf[0] & 0x1f) == 0x0C);
}

static void setup_for_ioctl(BIG_IOCTL_Command_struct *cmd,
		unsigned char *lunid,
                unsigned char *cdb, int cdblen,
                int write_direction,
                unsigned char *buf,
                int bufsize)
{
	memset(cmd, 0, sizeof(*cmd));
	memcpy(cmd->LUN_info.LunAddrBytes, lunid, 8);
	cmd->Request.CDBLen = cdblen;
	cmd->Request.Type.Type = TYPE_CMD;
	cmd->Request.Type.Attribute = ATTR_SIMPLE;
	cmd->Request.Type.Direction = (write_direction) ? XFER_WRITE : XFER_READ;
	cmd->Request.Timeout = 0;
	memcpy(cmd->Request.CDB, cdb, cdblen);
	cmd->buf_size = bufsize;
	cmd->buf = buf;
	cmd->malloc_size = 10000;
}

static int check_ioctl(char *filename, char *command,
		int rc, BIG_IOCTL_Command_struct *cmd)
{
	if (debug) {
		printf("CHK: %s: CommandStatus=%d\n", command,
			cmd->error_info.CommandStatus);
		if (cmd->error_info.CommandStatus == 1) {
			printf("CHK: ScsiStatus = %d\n", cmd->error_info.ScsiStatus);
		}
	}

	if (rc != 0) {
		fprintf(stderr, "%s: %s, %s ioctl failed, %s\n", progname,
			filename, command, strerror(errno));
		return -1;
	}
	if (cmd->error_info.CommandStatus != 0 && 
		cmd->error_info.CommandStatus != 2) {
		fprintf(stderr, "%s: %s%s, ioctl has Command Status=%d\n", 
			progname, filename, command, 
			cmd->error_info.CommandStatus);
		return -1;
	}
	return 0;
}

/* do cciss report physical luns to get list of physical device LUNIDs */
static int do_report_luns(char *filename, int fd,
	struct ReportLUNdata *lun, int *count)
{
	BIG_IOCTL_Command_struct cmd;
	int rc;
	unsigned char cdb[16];
	unsigned int bufsize;

	int *becount = (int *) lun;

	memset(lun, 0, sizeof(*lun));
	bufsize = htonl(sizeof(*lun));
	memset(cdb, 0, 16);
	cdb[0] = REPORT_PHYSICAL; /* report physical LUNs */
	memcpy(&cdb[6], &bufsize, 4);

	setup_for_ioctl(&cmd, CTLR_LUNID, cdb, 12, 0, (unsigned char *) lun, sizeof(*lun));
	rc = ioctl(fd, CCISS_BIG_PASSTHRU, &cmd);
	if (check_ioctl(filename, "report phys luns", rc, &cmd))
		return -1;
	*count = ntohl(*becount) / 8;
	return rc;
}

/* Send an inquiry for specified page to specified LUNID */
static int bmic_inquiry(char *filename, int fd, uint8_t *lunaddrbytes,
        uint8_t inquiry_page, uint8_t *buffer, uint8_t buffersize)
{
	int rc;
	unsigned char cdb[6];
	BIG_IOCTL_Command_struct bmic_command; 

	memset(cdb, 0, sizeof(cdb));
	cdb[0] = 0x12;
	cdb[1] = inquiry_page ? 0x01 : 0x00;
	cdb[2] = inquiry_page;
	cdb[4] = buffersize;

	setup_for_ioctl(&bmic_command, lunaddrbytes, cdb, sizeof(cdb),
			0, buffer, buffersize); 
	rc = ioctl(fd, CCISS_BIG_PASSTHRU, &bmic_command);
	if (check_ioctl(filename, "inquiry", rc, &bmic_command))
		return -1;
	
	return 0;
}

/* Check if device is suitable for changing to the specified mode
 *
 * If we are requesting OBDR mode, the device must be a tape drive
 * and must be from HP.
 *
 * If we are requesting to leave OBDR mode, the device must be
 * an OBDR device (have "$DR-10" at offset 43 in inquiry data
 */ 
static int is_suitable_device(char *filename, int fd, uint8_t *lunid, int mode)
{
	int rc;

#define OBDR_SIG_OFFSET 43
#define OBDR_SIGNATURE "$DR-10"
#define OBDR_SIG_LEN (sizeof(OBDR_SIGNATURE))
#define OBDR_TAPE_INQ_SIZE (OBDR_SIG_OFFSET + OBDR_SIG_LEN)

	unsigned char inqbuf[OBDR_TAPE_INQ_SIZE];
	unsigned char device_type;
	int is_obdr;

	/* Send inquiry to device. */
	rc = bmic_inquiry(filename, fd, lunid, 0, inqbuf, sizeof(inqbuf)); 
	if (rc) {
		if (debug)
			fprintf(stderr, "BMIC inquiry failed.\n");
		return 0;
	}

#define CD_DEVICE_TYPE 0x05
#define TAPE_DEVICE_TYPE 0x01
#define DEVICE_TYPE_MASK 0x1f
#define GET_DEVICE_TYPE(inquiry_data) ((inquiry_data)[0] & DEVICE_TYPE_MASK)

	/* Check for the "$DR-10" OBDR signature. */
	is_obdr = (strncmp((char *) &inqbuf[OBDR_SIG_OFFSET], OBDR_SIGNATURE, OBDR_SIG_LEN) == 0);
	if ((debug) || (mode == QUERY))
		fprintf(stderr, "Device at lunid "
				"0x%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx" 
				" is%s an OBDR device.\n", 
				lunid[0], lunid[1], lunid[2], lunid[3],
				lunid[4], lunid[5], lunid[6], lunid[7],
				is_obdr ?  "" : " not");
	if (!is_obdr)
		return 0;

	device_type = GET_DEVICE_TYPE(inqbuf);

	if ((debug) || (mode == QUERY)) {
		if (device_type == CD_DEVICE_TYPE) {
			fprintf(stderr, "Device at lunid "
					"0x%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx" 
					" is in CD-ROM mode (%02x).\n", 
					lunid[0], lunid[1], lunid[2], lunid[3],
					lunid[4], lunid[5], lunid[6], lunid[7],
					device_type);
		} else if (device_type == TAPE_DEVICE_TYPE) {
			fprintf(stderr, "Device at lunid "
					"0x%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx" 
					" is in Sequential mode (%02x).\n", 
					lunid[0], lunid[1], lunid[2], lunid[3],
					lunid[4], lunid[5], lunid[6], lunid[7],
					device_type);
		} else {
			fprintf(stderr, "Device at lunid "
					"0x%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx" 
					" is in an UNKNOWN mode (%02x).\n", 
					lunid[0], lunid[1], lunid[2], lunid[3],
					lunid[4], lunid[5], lunid[6], lunid[7],
					device_type);
		}
		if (mode == QUERY) /* don't exit if just debug... */
			return 0;
	}
			
	/* If we are requesting leave OBDR, check if "$DR-10" is at offset 43 */
	if (mode == TAPEMODE) {
		if (device_type != CD_DEVICE_TYPE)
			return 0;
		return 1;
	}
	/* else mode must be requesting CDMODE  */

	/* if the device is not a tape drive, bail... */
	if (device_type != TAPE_DEVICE_TYPE)
		return 0;

	/* else device is an HP tape drive. Proceed. */
	return 1;
}

/* Send mode select to device to switch it into OBDR mode */
static int set_obdr_mode(char *filename, int fd, uint8_t *lunid, int mode)
{

	BIG_IOCTL_Command_struct bmic_command; 
	unsigned char cdb[] = {0x15, 0x10, 0x00, 0x00, 0x08, 0x00};
	unsigned char enter_obdr_data[] = { 0x00, 0x00, 0x10, 0x00, 0x3e, 0x02, 0x01, 0x00 };
	unsigned char leave_obdr_data[] = { 0x00, 0x00, 0x00, 0x00, 0x3e, 0x02, 0x00, 0x00 };
	unsigned char *data;
	int rc;

	if (mode == QUERY)
		return 0;

	data = (mode == TAPEMODE) ? leave_obdr_data : enter_obdr_data;

	if (debug)
		fprintf(stderr, "Setting device to %s mode.\n", mode == TAPEMODE ? "tape" : "cd");

	setup_for_ioctl(&bmic_command, lunid, cdb, sizeof(cdb), 1,
				data, sizeof(enter_obdr_data)); 
	rc = ioctl(fd, CCISS_BIG_PASSTHRU, &bmic_command);
	if (check_ioctl(filename, "mode select", rc, &bmic_command))
		return -1;

	if (debug)
		fprintf(stderr, "Set mode to %s successfully\n", mode == TAPEMODE ? "tape" : "cd");
	return 0;
}

static int set_obdr_devices_mode(char *filename, int fd, int mode, char *mylunid)
{
	struct ReportLUNdata lun;
	int rc, i, count;
	uint8_t *lunid;
	char *foundlunid = NULL;

	/* Get a list of lunids for all physical devices on this smart array */
	count = sizeof(lun);
	rc = do_report_luns(filename, fd, &lun, &count);
	if (rc != 0) {
		if (debug)
			fprintf(stderr, "Report LUNs failed.\n");
		return rc;
	}

	if (debug)
		fprintf(stderr, "%d physical luns found.\n", count);
	/* For each physical device on this smart array ... */
	for (i = 0; i < count; i++) {
		lunid = &lun.LUN[i * 8];
		asprintf(&foundlunid, "0x%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx",
			lunid[0], lunid[1], lunid[2], lunid[3],
			lunid[4], lunid[5], lunid[6], lunid[7]);
		/* I passed a lunid and will only act for this one, so skip others */
		if ((mylunid) && (strcmp(mylunid, foundlunid) != 0)) {
			continue;
		}
		if (debug)
			fprintf(stderr, "Querying device at lunid %s\n", foundlunid);
		if (is_suitable_device(filename, fd, lunid, mode))
			set_obdr_mode(filename, fd, lunid, mode);
		free(foundlunid);
	}
	return 0;
}

static void usage(void)
{
	fprintf(stderr, "usage: %s [-vV] -m [tape|cd|query] [-l lunid] smart_array_device ...\n", progname);
	fprintf(stderr, "example: %s -m tape /dev/cciss/c1d0 ... to put the OBDR tape drive attached\n"
			"         to the second controller in tape mode\n", progname);
	exit(1);
}

int main(int argc, char *argv[])
{
	int i = 0, fd, mode = QUERY, opt;
	char *mylunid = NULL;
	static struct option long_options[] = {
                   {"verbose", 0, 0, 'v'},
                   {"lunid", 1, 0, 'l'},
                   {"mode", 1, 0, 'm'},
                   {"version", 0, 0, 'V'},
                   {0, 0, 0, 0},
	};


	progname = argv[0];

	if (argc < 2)
		usage();

	while ((opt = getopt_long(argc, argv, "Vvl:m:", long_options, NULL)) != -1) {
		i++;
		switch (opt) {
		case 'v':
			debug = 1;
			break;
		case 'l':
			asprintf(&mylunid, "%s", optarg);
			break;
		case 'm':
			if (strcmp(optarg, "tape") == 0)
				mode = TAPEMODE;
			else if (strcmp(optarg, "query") == 0)
				mode = QUERY;
			else if (strcmp(optarg, "cd") == 0)
				mode = CDMODE;
			else
				usage();
			break;
		case 'V':
			printf("%s version %s\n", argv[0], VERSION);
			exit(0);
		default: /* '?' */
			usage();
		}
	}

	if (optind >= argc) {
		fprintf(stderr, "Expected argument after options\n");
		usage();
	}

	for (i = optind ; i < argc; i++) {
		if (debug)
			fprintf(stderr, "%s:\n", argv[i]);
		fd = open(argv[i], O_RDWR);
		if (fd < 0) {
			fprintf(stderr, "%s: Can't open '%s': %s\n", 
				progname, argv[i], strerror(errno));
			continue;
		}
		if (!is_smartarray(fd)) {
			close(fd);
			continue;
		}
		set_obdr_devices_mode(argv[i], fd, mode, mylunid);
		close(fd);
	}
	free(mylunid);
	return 0;
}
