/***************************************************************************
$Id: mondo-rstr-tools.c 3893 2024-03-13 15:49:30Z bruno $
***************************************************************************/

#include <pthread.h>
#include <linux/fd.h>
#include "my-stuff.h"
#include "mr_mem.h"
#include "mr_str.h"
#include "mr_file.h"
#include "mr_sys.h"
#include "mondostructures.h"
#include "libmondo-mountlist-EXT.h"
#include "libmondo-tools-EXT.h"
#include "libmondo-filelist-EXT.h"
#include "libmondo-files-EXT.h"
#include "libmondo-devices-EXT.h"
#include "libmondo-string-EXT.h"
#include "libmondo-raid-EXT.h"
#include "libmondo-fork-EXT.h"
#include "newt-specific-EXT.h"
// no include for now in mondo-rstr-newt.c
extern int edit_filelist(struct s_node *);

extern struct s_bkpinfo *bkpinfo;

/**
 * The biggielist stub (appended to the directory where all.tar.gz was unpacked).
 */
#define BIGGIELIST_TXT_STUB "tmp/biggielist.txt"

/**
 * The filelist stub (appended to the directory where all.tar.gz was unpacked).
 */
#define FILELIST_FULL_STUB "tmp/filelist.full.gz"

/**
 * The mountlist stub (appended to the directory where all.tar.gz was unpacked).
 */
#define MOUNTLIST_FNAME_STUB "tmp/mountlist.txt"

/**
 * The mondorestore.cfg stub (appended to the directory where all.tar.gz was unpacked).
 */
#define MONDO_CFG_FILE_STUB "tmp/mondorestore.cfg"
/**
 * The i-want-my-lvm stub
 */
#define IWANTMYLVM_STUB "tmp/i-want-my-lvm"

extern bool g_ISO_restore_mode;	/* are we in Iso Mode? */
extern bool g_I_have_just_nuked;
/*
extern char *g_tmpfs_mountpt;
*/

/**
 * The device to mount to get at the ISO images. Ignored unless @p g_ISO_restore_mode.
 */
char *g_isodir_device;


/**
 * The format of @p g_isodir_device. Ignored unless @p g_ISO_restore_mode.
 */
char *g_isodir_format;

extern long g_current_progress, g_maximum_progress;
extern char *g_biggielist_txt;	// where 'biggielist.txt' is stored, on ramdisk / tempdir;
				  // biggielist.txt is the list of big files stored on the
				  // backup media set in question
extern char *g_filelist_full;	// filelist.full.gz is the list of all regular files
				  // (excluding big files) stored on the backup media set
extern char *g_biggielist_pot;	// list of big files which _could_ be restored, if the
				  // user chooses them
extern char *g_filelist_imagedevs;	// list of devices (e.g. /dev/hda1, /dev/sda5) which
					 // were archived as images, not just /dev entries
					 // ... e.g. NTFS, BeOS partitions
extern char *g_imagedevs_restthese;	// of the imagedevs listed in FILELIST_IMAGEDEVS,
					  // restore only these
extern char *g_mondo_cfg_file;	// where mondorestore.cfg (the config file) is stored
extern char *g_mountlist_fname;	// where mountlist.txt (the mountlist file) is stored
extern char *g_mondo_home;		// homedir of Mondo; usually /usr/local/share/mondo
extern int g_currentY;
extern bool g_restoring_live_from_cd;
extern bool g_text_mode;
extern int g_current_media_number;

extern t_bkptype g_backup_media_type;

extern int g_partition_table_locked_up;
extern char *MONDO_LOGFILE;

/* Should we use or not extended attributes and acl when restoring */
extern char *g_getfattr;
extern char *g_getfacl;

extern void kill_anything_like_this(char *str);
extern int skip_obdr(void);
extern int set_tape_block_size_with_mt(long internal_tape_block_size);

/**
* @addtogroup restoreUtilityGroup
* @{
*/
/**
* Free the malloc()s for the filename variables.
*/
void free_MR_global_filenames(void)
{
paranoid_free(g_biggielist_txt);
paranoid_free(g_filelist_full);
paranoid_free(g_filelist_imagedevs);
paranoid_free(g_imagedevs_restthese);
paranoid_free(g_mondo_cfg_file);
paranoid_free(g_mountlist_fname);
paranoid_free(g_mondo_home);
/*
paranoid_free(g_tmpfs_mountpt);
*/
paranoid_free(g_isodir_device);
paranoid_free(g_isodir_format);

}



/**
* Ask the user which imagedevs from the list contained in @p infname should
* actually be restored.
* @param infname The file containing a list of all imagedevs.
* @param outfname The location of the output file containing the imagedevs the user wanted to restore.
* @ingroup restoreUtilityGroup
*/
void ask_about_these_imagedevs(char *infname, char *outfname) {

FILE *fin;
FILE *fout;
char *incoming = NULL;
char *p = NULL;
char *question = NULL;

assert_string_is_neither_NULL_nor_zerolength(infname);
assert_string_is_neither_NULL_nor_zerolength(outfname);

if (!(fin = fopen(infname, "r"))) {
	fatal_error("Cannot openin infname");
}
if (!(fout = fopen(outfname, "w"))) {
	fatal_error("Cannot openin outfname");
}
for (mr_getline(p, fin); !feof(fin) && (p != NULL); mr_getline(p, fin)) {
	incoming = mr_strip_spaces(p);
	mr_free(p);

	if (incoming[0] == '\0') {
		mr_free(incoming);
		continue;
	}

	mr_asprintf(question, "Should I restore the image of %s ?", incoming);
	
	if (ask_me_yes_or_no(question)) {
		fprintf(fout, "%s\n", incoming);
	}
	mr_free(incoming);
}
mr_free(incoming);
mr_free(question);
paranoid_fclose(fout);
paranoid_fclose(fin);
}

/**************************************************************************
*ASK_ABOUT_THESE_IMAGEDEVS                                               *
**************************************************************************/



/**
 * Extract mondorestore.cfg and the mountlist from the tape inserted
 * to the ./tmp/ directory.
 * @param dev The tape device to read from.
 * @return 0 for success, nonzero for failure.
 */
int extract_cfg_file_and_mountlist_from_tape_dev(char *dev)
{
	char *command = NULL;
	int res = 0;

	if (bkpinfo->use_obdr) {
		skip_obdr();
	} else {
		// TODO: below 32KB seems to block at least on RHAS 2.1 and MDK 10.0
		set_tape_block_size_with_mt(bkpinfo->internal_tape_block_size);
	}

	mr_asprintf(command, "dd if=%s bs=%ld count=%ld 2> /dev/null | tar -zx ./%s ./%s ./%s ./%s ./%s", dev, bkpinfo->internal_tape_block_size, 1024L * 1024 * 32 / bkpinfo->internal_tape_block_size, MOUNTLIST_FNAME_STUB, MONDO_CFG_FILE_STUB, BIGGIELIST_TXT_STUB, FILELIST_FULL_STUB, IWANTMYLVM_STUB);
	log_msg(2, "command = '%s'", command);
	res = run_program_and_log_output(command, -1);
	mr_free(command);

	if (res != 0) {
		if (does_file_exist(MONDO_CFG_FILE_STUB)) {
			res = 0;
		} else {
			/* Doing that allow us to remain compatible with pre-2.2.5 versions */
			log_msg(2, "pre-2.2.4 compatible mode on");
			mr_asprintf(command,	"dd if=%s bs=%ld count=%ld 2> /dev/null | tar -zx %s %s %s %s %s", dev, bkpinfo->internal_tape_block_size, 1024L * 1024 * 32 / bkpinfo->internal_tape_block_size, MOUNTLIST_FNAME_STUB, MONDO_CFG_FILE_STUB, BIGGIELIST_TXT_STUB, FILELIST_FULL_STUB, IWANTMYLVM_STUB);
			log_msg(2, "command = '%s'", command);
			res = run_program_and_log_output(command, -1);
			mr_free(command);
			if ((res != 0) && (does_file_exist(MONDO_CFG_FILE_STUB))) {
				res = 0;
			}
		}
	}
	paranoid_free(command);
	return (res);
}





/**
 * Get the configuration file from the tape, or CD.
 * @param bkpinfo The backup information structure. Fields used:
 * - @c bkpinfo->backup_media_type
 * - @c bkpinfo->media_device
 * - @c bkpinfo->tmpdir
 * @return 0 for success, nonzero for failure.
 */
int get_cfg_file_from_archive()
{
	int retval = 0;

   /** malloc *****/
	char *command = NULL;
	char *cfg_file = NULL;
	char *tmp = NULL;
	char *tmp1 = NULL;
	char *mountpt = NULL;
	char *mountlist_file = NULL;
	bool extract_mountlist_stub = FALSE;
	bool extract_i_want_my_lvm = FALSE;

	bool try_plan_B;

	assert(bkpinfo != NULL);
	log_msg(2, "gcffa --- starting");
	log_to_screen("I'm thinking...");
	mr_asprintf(mountpt, "%s/mount.bootdisk", bkpinfo->tmpdir);
	if (chdir(bkpinfo->tmpdir)) {
		// FIXME
	}
	mr_asprintf(cfg_file, "%s", MONDO_CFG_FILE_STUB);
	unlink(cfg_file);			// cfg_file[] is missing the '/' at the start, FYI, by intent
	mr_free(cfg_file);

	unlink(FILELIST_FULL_STUB);
	unlink(BIGGIELIST_TXT_STUB);
	mr_asprintf(command, "mkdir -p %s", mountpt);
	run_program_and_log_output(command, FALSE);
	mr_free(command);

	mr_asprintf(cfg_file, "%s/%s", bkpinfo->tmpdir, MONDO_CFG_FILE_STUB);
	mr_asprintf(mountlist_file, "%s/%s", bkpinfo->tmpdir, MOUNTLIST_FNAME_STUB);
	log_msg(2, "mountpt = %s; cfg_file=%s", mountpt, cfg_file);
	mr_free(mountpt);

	if (!does_file_exist(cfg_file)) {
		log_msg(2, "gcffa --- we don't have cfg file yet.");
		if (IS_THIS_A_STREAMING_BACKUP(bkpinfo->backup_media_type)) {
			try_plan_B = TRUE;
		} else {
			log_msg(2, "gcffa --- calling mount_media now :)");
			if (!mount_media(MNT_CDROM)) {
				log_msg(2, "gcffa --- managed to mount CD; so, no need for Plan B");
				try_plan_B = FALSE;
			} else {
				try_plan_B = TRUE;
			}
			if (what_number_cd_is_this() > 1) {
				insist_on_this_cd_number((g_current_media_number = 1));
			}
		}
		if (try_plan_B) {
			log_msg(2, "gcffa --- OK, switching to Plan B");
			if (chdir(bkpinfo->tmpdir)) {
				// FIXME
			}
			run_program_and_log_output("mkdir -p tmp", FALSE);

			if (bkpinfo->media_device == NULL) {
				mr_asprintf(bkpinfo->media_device, "%s", "/dev/st0");
				log_msg(2, "media_device is blank; assuming %s", bkpinfo->media_device);
			}
			mr_asprintf(tmp, "%s", bkpinfo->media_device);
			if (extract_cfg_file_and_mountlist_from_tape_dev(bkpinfo->media_device)) {
				mr_free(bkpinfo->media_device);
				mr_asprintf(bkpinfo->media_device, "%s", "/dev/st0");
				if (extract_cfg_file_and_mountlist_from_tape_dev(bkpinfo->media_device)) {
					mr_free(bkpinfo->media_device);
					mr_asprintf(bkpinfo->media_device, "%s", "/dev/osst0");
					if (extract_cfg_file_and_mountlist_from_tape_dev(bkpinfo->media_device)) {
						mr_free(bkpinfo->media_device);
						mr_asprintf(bkpinfo->media_device, "%s", "/dev/ht0");
						if (extract_cfg_file_and_mountlist_from_tape_dev(bkpinfo->media_device)) {
							log_msg(3, "I tried lots of devices but none worked.");
							mr_free(bkpinfo->media_device);
						}
					}
				}
			}
			if (bkpinfo->media_device == NULL) {
				bkpinfo->media_device = tmp;
			} else {
				mr_free(tmp);
			}

			if (!does_file_exist("tmp/mondorestore.cfg")) {
				log_to_screen("Cannot find config info on media");
				return (1);
			}
		} else {
				if (does_file_exist("/"MOUNTLIST_FNAME_STUB)) {
					extract_mountlist_stub = FALSE;
				} else {
					extract_mountlist_stub = TRUE;
				}
				if (does_file_exist("/"IWANTMYLVM_STUB)) {
					extract_i_want_my_lvm = FALSE;
				} else {
					extract_i_want_my_lvm = TRUE;
				}

				log_msg(2, "gcffa --- Plan B, a.k.a. untarring some file from all.tar.gz");
				mr_asprintf(command, "tar -zxvf " MNT_CDROM "/images/all.tar.gz ./%s ./%s ./%s ./%s ./%s", MOUNTLIST_FNAME_STUB, MONDO_CFG_FILE_STUB, BIGGIELIST_TXT_STUB, FILELIST_FULL_STUB, IWANTMYLVM_STUB);	// add -b TAPE_BLOCK_SIZE if you _really_ think it's necessary
				run_program_and_log_output(command, TRUE);
				mr_free(command);

				if (!does_file_exist(MONDO_CFG_FILE_STUB)) {
					/* Doing that allow us to remain compatible with pre-2.2.5 versions */
					log_msg(2, "pre-2.2.4 compatible mode on");
					mr_asprintf(command, "tar -zxvf " MNT_CDROM "/images/all.tar.gz %s %s %s %s %s", MOUNTLIST_FNAME_STUB, MONDO_CFG_FILE_STUB, BIGGIELIST_TXT_STUB, FILELIST_FULL_STUB, IWANTMYLVM_STUB);	// add -b TAPE_BLOCK_SIZE if you _really_ think it's necessary
					run_program_and_log_output(command, TRUE);
					mr_free(command);
					if (!does_file_exist(MONDO_CFG_FILE_STUB)) {
						fatal_error
							("Please reinsert the disk/CD and try again.");
					}
				}
		}
	}
	if (does_file_exist(MONDO_CFG_FILE_STUB)) {
		log_msg(1, "gcffa --- great! We've got the config file");
		tmp1 = call_program_and_get_last_line_of_output("pwd");
		mr_asprintf(tmp, "%s/%s", tmp1, MONDO_CFG_FILE_STUB);
		mr_asprintf(command, "cp -f %s %s", tmp, cfg_file);
		log_it("%s",command);
		if (strcmp(tmp, cfg_file) && run_program_and_log_output(command, 1)) {
			log_msg(1,
					"... but an error occurred when I tried to move it to %s",
					cfg_file);
		} else {
			log_msg(1, "... and I moved it successfully to %s", cfg_file);
		}
		mr_free(command);

		mr_asprintf(command, "cp -f %s/%s %s", tmp1, MOUNTLIST_FNAME_STUB, mountlist_file);
		mr_free(tmp1);
		log_it("%s",command);
		if (extract_mountlist_stub) {
			if (strcmp(tmp, cfg_file) && run_program_and_log_output(command, 1)) {
				log_msg(1, "Failed to get mountlist");
			} else {
				log_msg(1, "Got mountlist too");
				mr_free(command);
				mr_asprintf(command, "cp -f %s %s", mountlist_file,	g_mountlist_fname);
				if (run_program_and_log_output(command, 1)) {
					log_msg(1, "Failed to copy mountlist to /tmp");
				} else {
					log_msg(1, "Copied mountlist to /tmp as well OK");
					mr_free(command);
					mr_asprintf(command, "cp -f %s /tmp/",IWANTMYLVM_STUB);
					run_program_and_log_output(command, 1);
				}
			}
		}
		mr_free(command);
		mr_free(tmp);
	}

	run_program_and_log_output("umount -d " MNT_CDROM, FALSE);
	if (!does_file_exist(cfg_file)) {
		log_it("%s",cfg_file);
		log_msg(1, "%s not found", cfg_file);
		log_to_screen("Oh dear. Unable to recover configuration file from boot disk");
		return (1);
	}

	log_to_screen("Recovered mondorestore.cfg");
	if (!does_file_exist(MOUNTLIST_FNAME_STUB)) {
		log_to_screen("...but not mountlist.txt - a pity, really...");
	} else {
			/* Is this code really useful ??? */
		if (extract_mountlist_stub) {
			mr_asprintf(command, "cp -f %s %s/%s", MOUNTLIST_FNAME_STUB, bkpinfo->tmpdir, MOUNTLIST_FNAME_STUB);
			run_program_and_log_output(command, FALSE);
			mr_free(command);
		}
	}

	mr_asprintf(command, "cp -f %s /%s", cfg_file, MONDO_CFG_FILE_STUB);
	mr_free(cfg_file);

	run_program_and_log_output(command, FALSE);
	mr_free(command);

	if (extract_mountlist_stub) {
		mr_asprintf(command, "cp -f %s /%s", mountlist_file, MOUNTLIST_FNAME_STUB);
		run_program_and_log_output(command, FALSE);
		mr_free(command);
	}
	mr_free(mountlist_file);

	mr_asprintf(command, "cp -f etc/raidtab /etc/");
	run_program_and_log_output(command, FALSE);
	mr_free(command);

	if (extract_i_want_my_lvm) {
		mr_asprintf(command, "cp -f %s /tmp/",IWANTMYLVM_STUB);
		run_program_and_log_output(command, FALSE);
		mr_free(command);
	}
	g_backup_media_type = bkpinfo->backup_media_type;
	return (retval);
}

/**************************************************************************
 *END_GET_CFG_FILE_FROM_ARCHIVE                                           *
 **************************************************************************/




/**
* Extract @c mondorestore.cfg and @c mountlist.txt from @p ramdisk_fname.
* @param bkpinfo The backup information structure. @c tmpdir is the only field used.
* @param ramdisk_fname The filename of the @b compressed ramdisk to look in.
* @param output_cfg_file Where to put the configuration file extracted.
* @param output_mountlist_file Where to put the mountlist file extracted.
* @return 0 for success, nonzero for failure.
* @ingroup restoreUtilityGroup
*/

/**
* Keep trying to get mondorestore.cfg from the archive, until the user gives up.
*/
void get_cfg_file_from_archive_or_bust()
{
while (get_cfg_file_from_archive()) {
	if (!ask_me_yes_or_no("Failed to find config file/archives. Choose another source?")) {
		fatal_error("Could not find config file/archives. Aborting.");
	}
	interactively_obtain_media_parameters_from_user(FALSE);
	}
}


/**
* Determine whether @p list_fname contains a line containing @p f.
* @param f The line to search for.
* @param list_fname The file to search in.
* @param preamble Ignore this beginning part of @p f ("" to disable).
* @return TRUE if it's in the list, FALSE if it's not.
*/
bool is_file_in_list(char *f, char *list_fname, char *preamble)
{

char *command = NULL;
char *file = NULL;
char *tmp = NULL;
int res;

assert_string_is_neither_NULL_nor_zerolength(f);
assert_string_is_neither_NULL_nor_zerolength(list_fname);
assert(preamble != NULL);

if (strncmp(preamble, f, strlen(preamble)) == 0) {
	mr_asprintf(file, "%s", f + strlen(preamble));
} else {
	mr_asprintf(file, "%s", f);
}
if (file[0] == '/' && file[1] == '/') {
	tmp = file;

	mr_asprintf(file, "%s", tmp + 1);
	mr_free(tmp);
}
log_msg(2, "Checking to see if f=%s, file=%s, is in the list of biggiefiles", f, file);
mr_asprintf(command, "grep -E '^%s$' %s", file, list_fname);
mr_free(file);

res = run_program_and_log_output(command, FALSE);
mr_free(command);
if (res) {
	return (FALSE);
} else {
	return (TRUE);
}
}

/**************************************************************************
*END_IS_FILE_IN_LIST                                                     *
**************************************************************************/

/**
 * Get information about the location of ISO images from the user.
 * @param isodir_device Where to put the device (e.g. /dev/hda4) the user enters.
 * @param isodir_format Where to put the format (e.g. ext2) the user enters.
 * @param nuke_me_please Whether we're planning on nuking or not.
 * @return TRUE if OK was pressed, FALSE otherwise.
 */
bool get_isodir_info(char *isodir_device, char *isodir_format, bool nuke_me_please)
{

	bool HDD = FALSE;
	char *tmp = NULL;
	/** initialize ********************************************************/

	assert(isodir_device != NULL);
	assert(isodir_format != NULL);
	assert(bkpinfo->isodir != NULL);

	log_it("bkpinfo->isodir = %s", bkpinfo->isodir);
	isodir_format[0] = '\0';
	if (isodir_device[0] == '\0') {
		strcpy(isodir_device, "/dev/");
	}
	if (bkpinfo->isodir == NULL) {
		mr_asprintf(bkpinfo->isodir, "%s", "/");
	}
	// TODO: Are we sure it does what it's expected to do here ?
	if (does_file_exist("/tmp/NETFS-SERVER-MOUNT")) {
		strcpy(isodir_device, last_line_of_file("/tmp/NETFS-SERVER-MOUNT"));
		strcpy(isodir_format, "netfs");
		mr_asprintf(bkpinfo->isodir, "%s", last_line_of_file("/tmp/ISO-DIR"));
	}
	if (nuke_me_please) {
		return (TRUE);
	}

	if (popup_and_get_string("ISO Mode - device", "On what device do the ISO files live?", isodir_device, MAX_STR_LEN / 4)) {
		if (popup_and_get_string ("ISO Mode - format", "What is the disk format of the device? (Hit ENTER if you don't know.)", isodir_format, 16)) {
			if (popup_with_buttons("Are you restoring from an external hard drive ?", "Yes", "No")) {
				HDD = TRUE;
			}
			tmp = mr_popup_and_get_string ("ISO Mode - path", "At what path on this device can the ISO files be found ?", bkpinfo->isodir);
			if (tmp) {
				strip_spaces(isodir_device);
				strip_spaces(isodir_format);
				bkpinfo->isodir = mr_strip_spaces(tmp);
				mr_free(tmp);
				log_it("isodir_device = %s - isodir_format = %s - bkpinfo->isodir = %s", isodir_device, isodir_format, bkpinfo->isodir);
				if (HDD) {
					/*  We need an additional param */
					mr_asprintf(bkpinfo->subdir, "%s", bkpinfo->isodir);
					mr_asprintf(bkpinfo->isodir, "%s", "/tmp/isodir");
					log_it("Now bkpinfo->isodir = %s and subdir = ", bkpinfo->isodir, bkpinfo->subdir);
				}
				return (TRUE);
			}
		}
	}
	return (FALSE);
}


/**
* Set up an ISO backup.
* @param bkpinfo The backup information structure. Fields used:
* - @c bkpinfo->backup_media_type
* - @c bkpinfo->disaster_recovery
* - @c bkpinfo->isodir
* @param nuke_me_please If TRUE, we're in nuke mode; if FALSE we're in interactive mode.
* @return 0 for success, nonzero for failure.
*/
int iso_fiddly_bits(bool nuke_me_please) {

char *mount_isodir_command = NULL;
char *command = NULL;
char *mds = NULL;
int retval = 0, i;

g_ISO_restore_mode = TRUE;
read_cfg_var(g_mondo_cfg_file, "iso-dev", g_isodir_device);
if (bkpinfo->disaster_recovery) {
	/* Don't let this clobber an existing bkpinfo->isodir */
	if (bkpinfo->isodir == NULL) {
		mr_asprintf(bkpinfo->isodir, "%s", "/tmp/isodir");
	}
	mr_asprintf(command, "mkdir -p %s", bkpinfo->isodir);
	run_program_and_log_output(command, 5);
	mr_free(command);
	log_msg(2, "Setting isodir to %s", bkpinfo->isodir);
}

if (!get_isodir_info(g_isodir_device, g_isodir_format, nuke_me_please)) {
	return (1);
}
paranoid_system("umount -d " MNT_CDROM " 2> /dev/null");	/* just in case */

if (is_this_device_mounted(g_isodir_device)) {
	log_to_screen("WARNING - isodir is already mounted");
} else {
	mr_asprintf(mount_isodir_command, "mount %s", g_isodir_device);
	if (strlen(g_isodir_format) > 1) {
		mr_strcat(mount_isodir_command, " -t %s", g_isodir_format);
	}

	mr_strcat(mount_isodir_command, " -o ro %s", bkpinfo->isodir);
	run_program_and_log_output("df -m -P -T", FALSE);
	log_msg(1, "The 'mount' command is '%s'. PLEASE report this command to be if you have problems, ok?", mount_isodir_command);
	if (run_program_and_log_output(mount_isodir_command, FALSE)) {
		popup_and_OK("Cannot mount the device where the ISO files are stored.");
		return (1);
	}
	paranoid_free(mount_isodir_command);
	log_to_screen("I have mounted the device where the ISO files are stored.");
}
if (!IS_THIS_A_STREAMING_BACKUP(bkpinfo->backup_media_type)) {
	mount_media(MNT_CDROM);
}
i = what_number_cd_is_this();	/* has the side-effect of calling mount_media() */
mds = media_descriptor_string(bkpinfo->backup_media_type);
mr_free(mds);

log_msg(1, "%s #%d has been mounted via loopback mount", mds, i);
if (i < 0) {
	popup_and_OK("Cannot find ISO images in the directory you specified.");
	retval = 1;
}
log_msg(2, "bkpinfo->isodir is now %s", bkpinfo->isodir);
return (retval);
}




/**
* Kill all Petris processes.
*/
void kill_petris(void) {
	kill_anything_like_this("petris");
}

/**************************************************************************
*END_KILL_PETRIS                                                         *
**************************************************************************/


/**
 * Mount @p device at @p mpt as @p format.
 * @param device The device (/dev entry) to mount.
 * @param mpt The directory to mount it on.
 * @param format The filesystem type of @p device.
 * @param writeable If TRUE, mount read-write; if FALSE, mount read-only.
 * @return 0 for success, nonzero for failure.
 */
int mount_device(char *device, char *mpt, char *format, bool writeable)
{
int res = 0;

/** malloc **/
char *tmp = NULL;
char *command = NULL;
char *mountdir = NULL;
char *mountpoint = NULL;
char *additional_parameters = NULL;

assert_string_is_neither_NULL_nor_zerolength(device);
assert_string_is_neither_NULL_nor_zerolength(mpt);
assert(format != NULL);

	if (!strcmp(mpt, "/1")) {
		mr_asprintf(mountpoint, "/");
		log_msg(3, "Mommm! SME is being a dildo!");
	} else {
		mr_asprintf(mountpoint, "%s", mpt);
	}

	if (!strcmp(mountpoint, "lvm")) {
		return (0);
	}
	if (!strcmp(mountpoint, "image")) {
		return (0);
	}
	mr_asprintf(tmp, "Mounting device %s   ", device);
	log_msg(1, tmp);
	/* Deal with additional params only if not /proc or /sys */
	mr_asprintf(additional_parameters, "");
	if (strcmp(format, "proc") && strcmp(format, "sys")) {
		if (writeable) {
			mr_strcat(additional_parameters, "-o rw");
		} else {
			mr_strcat(additional_parameters, "-o ro");
		}
		if ((tmp = find_home_of_exe("setfattr")) != NULL) {
			mr_strcat(additional_parameters, ",user_xattr");
		}
		mr_free(tmp);
		if ((tmp = find_home_of_exe("setfacl")) != NULL) {
			mr_strcat(additional_parameters, ",acl");
		}
		mr_free(tmp);
	}

	if (!strcmp(mountpoint, "swap")) {
		mr_asprintf(command, "swapon %s", device);
		mr_asprintf(mountdir, "swap");
	} else {
		if (!strcmp(mountpoint, "/")) {
			mr_asprintf(mountdir, "%s", MNT_RESTORING);
		} else {
			mr_asprintf(mountdir, "%s%s", MNT_RESTORING, mountpoint);
		}
		mr_asprintf(command, "mkdir -p %s", mountdir);
		run_program_and_log_output(command, FALSE);
		mr_free(command);

		mr_asprintf(command, "mount -t %s %s %s %s 2>> %s", format, device, additional_parameters, mountdir, MONDO_LOGFILE);
		log_msg(2, "command='%s'", command);
	}
	mr_free(additional_parameters);

	res = run_program_and_log_output(command, TRUE);
	if (res && (strstr(command, "xattr") || strstr(command, "acl"))) {
		log_msg(1, "Re-trying without the fancy extra parameters");
		mr_free(command);
		mr_asprintf(command, "mount -t %s %s %s 2>> %s", format, device, mountdir, MONDO_LOGFILE);
		res = run_program_and_log_output(command, TRUE);
	}
	if (res) {
		log_msg(1, "Unable to mount device %s (type %s) at %s", device, format, mountdir);
		log_msg(1, "command was '%s'", command);
		if (!strcmp(mountpoint, "swap")) {
			log_to_screen(tmp);
		} else {
			log_msg(2, "Retrying w/o the '-t' switch");
			mr_free(command);
			mr_asprintf(command, "mount %s %s 2>> %s", device, mountdir, MONDO_LOGFILE);
			log_msg(2, "2nd command = '%s'", command);
			res = run_program_and_log_output(command, TRUE);
			if (res == 0) {
				log_msg(1,
					"That's OK. I called mount w/o a filesystem type and it worked fine in the end.");
			} else {
				log_to_screen(tmp);
			}
		}
	}

	if (res && !strcmp(mountpoint, "swap")) {
		log_msg(2, "That's ok. It's just a swap partition.");
		log_msg(2, "Non-fatal error. Returning 0.");
		res = 0;
	}

mr_free(tmp);
mr_free(command);
mr_free(mountdir);
mr_free(mountpoint);

return (res);
}
/**************************************************************************
 *END_MOUNT_DEVICE                                                        *
**************************************************************************/


/**
 * Mount all devices in @p p_external_copy_of_mountlist on @p MNT_RESTORING.
 * @param p_external_copy_of_mountlist The mountlist containing devices to be mounted.
 * @param writeable If TRUE, then mount read-write; if FALSE mount read-only.
 * @return The number of errors encountered (0 for success).
 */
int mount_all_devices(struct mountlist_itself *p_external_copy_of_mountlist, bool writeable)
{
int retval = 0, lino, res;
char *tmp = NULL;
char *these_failed = NULL;
struct mountlist_itself *mountlist = NULL;

assert(p_external_copy_of_mountlist != NULL);
mountlist = malloc(sizeof(struct mountlist_itself));
memcpy((void *) mountlist, (void *) p_external_copy_of_mountlist,
   sizeof(struct mountlist_itself));
	sort_mountlist_by_mountpoint(mountlist, 0);


	mvaddstr_and_log_it(g_currentY, 0, "Mounting devices         ");
	open_progress_form("Mounting devices", "I am now mounting all the drives.", "This should not take long.", "", mountlist->entries);

	mr_asprintf(these_failed, "");
	for (lino = 0; lino < mountlist->entries; lino++) {
		if (!strcmp(mountlist->el[lino].device, "/proc")) {
			log_msg(1, "Again with the /proc - why is this in your mountlist?");
		} else if (is_this_device_mounted(mountlist->el[lino].device)) {
			log_to_screen("%s is already mounted", mountlist->el[lino].device);
		} else if (strcmp(mountlist->el[lino].mountpoint, "none")
		   && strcmp(mountlist->el[lino].mountpoint, "lvm")
		   && strcmp(mountlist->el[lino].mountpoint, "raid")
		   && strcmp(mountlist->el[lino].mountpoint, "image")) {
			mr_asprintf(tmp, "Mounting %s", mountlist->el[lino].device);
			update_progress_form(tmp);
			mr_free(tmp);
			res = mount_device(mountlist->el[lino].device, mountlist->el[lino].mountpoint, mountlist->el[lino].format, writeable);
			retval += res;
			if (res) {
				mr_strcat(these_failed, "%s ",mountlist->el[lino].device);
			}
		}
		g_current_progress++;
	}
	close_progress_form();
	if (retval) {
		if (g_partition_table_locked_up > 0) {
			log_to_screen("fdisk's ictol() call to refresh its copy of the partition table causes the kernel to");
			log_to_screen("lock up the partition table. You might have to reboot and use Interactive Mode to");
			log_to_screen("format and restore *without* partitioning first. Sorry for the inconvenience.");
		}
		mr_asprintf(tmp, "Could not mount device(s) %s- shall I abort?", these_failed);

		if (!ask_me_yes_or_no(tmp)) {
			retval = 0;
			log_to_screen("Continuing, although some device(s) failed to be mounted");
			mvaddstr_and_log_it(g_currentY++, 74, "Done.");
		} else {
			mvaddstr_and_log_it(g_currentY++, 74, "Failed.");
			log_to_screen("Unable to mount some or all of your partitions.");
		}
		mr_free(tmp);
	} else {
		log_to_screen("All partitions were mounted OK.");
		mvaddstr_and_log_it(g_currentY++, 74, "Done.");
	}
	paranoid_free(these_failed);

	/* Also mounting under MNT_RESTORING  special FS */
	(void)mount_device("/proc","/proc","proc",TRUE);
	(void)mount_device("/sys","/sys","sysfs",TRUE);
	/*  BR#852: we need to create BTRFS subvols now */
	/*  Creating potential btrfs subvolumes */
	if (does_file_exist("/tmp/create-btrfs-subvol")) {
		run_program_and_log_output("/tmp/create-btrfs-subvol",3);
	}
	/*  Mounting potential btrfs subvolumes */
	if (does_file_exist("/tmp/mount-btrfs-subvol")) {
		run_program_and_log_output("/tmp/mount-btrfs-subvol",3);
	}
	run_program_and_log_output("df -m -P -T", 3);
	paranoid_free(mountlist);
	return (retval);
}
/**************************************************************************
*END_MOUNT_ALL_DEVICES                                                   *
**************************************************************************/


/**
* Fix some miscellaneous things in the filesystem so the system will come
* up correctly on the first boot.
*/
void protect_against_braindead_sysadmins() {

run_program_and_log_output("touch " MNT_RESTORING "/var/log/pacct", 0);
run_program_and_log_output("touch " MNT_RESTORING "/var/account/pacct", 0);

/* Log the status of the boot area */
run_program_and_log_output("ls -alR " MNT_RESTORING "/boot", 0);

if (run_program_and_log_output("ls -al " MNT_RESTORING " /tmp", 0)) {
	run_program_and_log_output("chmod 1777 " MNT_RESTORING "/tmp", 0);
}
run_program_and_log_output("mkdir -p " MNT_RESTORING "/var/run/console", 0);
run_program_and_log_output("chmod 666 " MNT_RESTORING "/dev/null", 0);
/* This doesn't work as the mkdir should be done in home ! Maybe time to remove 
   run_program_and_log_output("cd " MNT_RESTORING "; for i in `ls home/`; do echo \"Moving $i's spurious files to $i/.disabled\"; mkdir \"$i\"/.disabled ; mv -f \"$i\"/.DCOP* \"$i\"/.MCOP* \"$i\"/.*authority \"$i\"/.kde/tmp* \"$i\"/.kde/socket* \"$i\"/.disabled/ ; done", 1);
   */
run_program_and_log_output("rm -f " MNT_RESTORING "/var/run/*.pid", 1);
run_program_and_log_output("rm -f " MNT_RESTORING "/var/lock/subsys/*", 1);
}

/**************************************************************************
*END_PROTECT_AGAINST_BRAINDEAD_SYSADMINS                                 *
**************************************************************************/




/**
* Fill out @p bkpinfo based on @p cfg_file.
* @param cfg_file The mondorestore.cfg file to read into @p bkpinfo.
* @param bkpinfo The backup information structure to fill out with information
* from @p cfg_file.
* @return 0 for success, nonzero for failure.
*/
int read_cfg_file_into_bkpinfo(char *cfgf)
{
/** add mallocs **/
char value[MAX_STR_LEN];
char *tmp = NULL;
char *tmp1 = NULL;
char *envtmp1 = NULL;
char *envtmp2 = NULL;
char *command = NULL;
char iso_mnt[MAX_STR_LEN];
char iso_path[MAX_STR_LEN];
char *old_isodir = NULL;
char *cfg_file = NULL;
t_bkptype media_specified_by_user;

//  assert_string_is_neither_NULL_nor_zerolength(cfg_file);
assert(bkpinfo != NULL);
assert(cfgf != NULL);
log_it("Entering read_cfg_file_into_bkpinfo");

if (!cfgf) {
	mr_asprintf(cfg_file, "%s", g_mondo_cfg_file);
} else {
	mr_asprintf(cfg_file, "%s", cfgf);
}

media_specified_by_user = bkpinfo->backup_media_type;	// or 'none', if not specified

if (0 == read_cfg_var(cfg_file, "backup-media-type", value)) {
	if (!strcmp(value, "optical")) {
		bkpinfo->backup_media_type = optical;
	} else if (!strcmp(value, "usb")) {
		bkpinfo->backup_media_type = usb;
		bkpinfo->please_dont_eject = TRUE;
	} else if (!strcmp(value, "iso")) {
		bkpinfo->backup_media_type = iso;
		if (am_I_in_disaster_recovery_mode()) {
			/* Check to see if CD is already mounted before mounting it... */
			if (!is_this_device_mounted("/dev/cdrom")) {
				log_msg(2, "NB: CDROM device not mounted, mounting...");
				run_program_and_log_output("mount /dev/cdrom "MNT_CDROM, 1);
			}
			if (does_file_exist(ARCHIVES_PATH "/filelist.0")) {
				bkpinfo->backup_media_type = optical;
				run_program_and_log_output("umount -d "MNT_CDROM, 1);
				log_it("Re-jigging configuration AGAIN. Optical, not ISO.");
			}
		}
		if (read_cfg_var(cfg_file, "iso-prefix", value) == 0) {
			mr_asprintf(bkpinfo->prefix, "%s", value);
		} else {
			mr_asprintf(bkpinfo->prefix, "%s", STD_PREFIX);
		}
		log_it("Setting Prefix to %s", bkpinfo->prefix);
	} else if ((!strcmp(value, "netfs")) || (!strcmp(value, "nfs"))) {
		/* Stay compatible with previous versions by allowing nfs as an entry here */
		bkpinfo->backup_media_type = netfs;
		bkpinfo->please_dont_eject = TRUE;
		if (read_cfg_var(cfg_file, "netfs-proto", value) == 0) {
			mr_asprintf(bkpinfo->netfs_proto,"%s", value);
		} else {
			/* For compatibility, force protocol in old nfs case to be transparent */
			mr_asprintf(bkpinfo->netfs_proto, "nfs");
		}
		if (read_cfg_var(cfg_file, "netfs-server-user", value) == 0) {
			mr_asprintf(bkpinfo->netfs_user,"%s", value);
		}
		if (read_cfg_var(cfg_file, "iso-prefix", value) == 0) {
			mr_asprintf(bkpinfo->prefix, "%s", value);
		} else {
			mr_asprintf(bkpinfo->prefix, "%s", STD_PREFIX);
		}
		tmp1 = call_program_and_get_last_line_of_output("cat "CMDLINE);
		if (strstr(tmp1, "pxe")) {
			/* We need to override prefix value in PXE mode as it's
			* already done in start-netfs */
			envtmp1 = getenv("imgname");
			if (envtmp1 == NULL) {
				fatal_error("no imgname variable in environment");
			}
			mr_asprintf(bkpinfo->prefix, "%s", envtmp1);
		}	
		mr_free(tmp1);
	} else if (!strcmp(value, "tape")) {
		bkpinfo->backup_media_type = tape;
	} else if (!strcmp(value, "udev")) {
		bkpinfo->backup_media_type = udev;
	} else {
		fatal_error("UNKNOWN bkp-media-type");
	}
} else {
	fatal_error("backup-media-type not specified!");
}

if (bkpinfo->disaster_recovery) {
	if (bkpinfo->backup_media_type == usb) {
		envtmp1 = getenv("MRUSBDEV");
		if (envtmp1 == NULL) {
			if (read_cfg_var(cfg_file, "usb-dev", value)) {
				fatal_error("Cannot get USB device name from cfg file");
			}
		} else {
			strcpy(value,envtmp1);
		}
		mr_asprintf(bkpinfo->media_device, "%s1", value);
		log_msg(2, "Backup medium is USB --- dev=%s", bkpinfo->media_device);
	} else if (bkpinfo->backup_media_type == tape || bkpinfo->backup_media_type == udev) {
		if (read_cfg_var(cfg_file, "media-dev", value)) {
			fatal_error("Cannot get tape device name from cfg file");
		}
		mr_asprintf(bkpinfo->media_device, "%s", value);
		read_cfg_var(cfg_file, "media-size", value);
		bkpinfo->media_size = atol(value);
		log_msg(2, "Backup medium is TAPE --- dev=%s", bkpinfo->media_device);
	} else {
		mr_asprintf(bkpinfo->media_device, "%s", "/dev/cdrom");	/* we don't really need this var */
		bkpinfo->media_size = 1999 * 1024;	/* 650, probably, but we don't need this var anyway */
		log_msg(2, "Backup medium is similar to CD-R[W]");
	}
} else {
	log_msg(2, "Not in Disaster Recovery Mode. No need to derive device name from config file.");
}

read_cfg_var(cfg_file, "use-star", value);
if (strstr(value, "yes")) {
	bkpinfo->use_star = TRUE;
	log_msg(1, "Goody! ... bkpinfo->use_star is now true.");
}

read_cfg_var(cfg_file, "obdr", value);
if (strstr(value, "TRUE")) {
	bkpinfo->use_obdr = TRUE;
	log_msg(1, "OBDR mode activated");
}

read_cfg_var(cfg_file, "acl", value);
if (strstr(value, "TRUE")) {
	if ((tmp = find_home_of_exe("setfacl")) == NULL) {
		log_msg(1, "Unable to restore ACLs as no setfacl found");
	} else {
		mr_asprintf(g_getfacl,"%s", tmp);
		log_msg(1, "We will restore ACLs");
	}
	mr_free(tmp);
}
read_cfg_var(cfg_file, "xattr", value);
if (strstr(value, "TRUE")) {
	if ((tmp = find_home_of_exe("setfattr")) == NULL) {
		log_msg(1, "Unable to restore XATTRs as no setfattr found");
	} else {
		mr_asprintf(g_getfattr,"%s", tmp);
		log_msg(1, "We will restore XATTRs");
	}
	mr_free(tmp);
}

if (0 == read_cfg_var(cfg_file, "internal-tape-block-size", value)) {
	bkpinfo->internal_tape_block_size = atol(value);
} else {
	bkpinfo->internal_tape_block_size = DEFAULT_INTERNAL_TAPE_BLOCK_SIZE;
}
log_msg(1, "Internal tape block size set to %ld", bkpinfo->internal_tape_block_size);

read_cfg_var(cfg_file, "use-lzma", value);
if (strstr(value, "yes")) {
	bkpinfo->use_lzma = TRUE;
	bkpinfo->use_lzo = FALSE;
	bkpinfo->use_gzip = FALSE;
	mr_asprintf(bkpinfo->zip_exe, "%s", "lzma");
	mr_asprintf(bkpinfo->zip_suffix, "%s", "lzma");
}

read_cfg_var(cfg_file, "use-lzo", value);
if (strstr(value, "yes")) {
	bkpinfo->use_lzma = FALSE;
	bkpinfo->use_lzo = TRUE;
	bkpinfo->use_gzip = FALSE;
	mr_asprintf(bkpinfo->zip_exe, "%s", "lzop");
	mr_asprintf(bkpinfo->zip_suffix, "%s", "lzo");
}

read_cfg_var(cfg_file, "use-gzip", value);
if (strstr(value, "yes")) {
	bkpinfo->use_lzma = FALSE;
	bkpinfo->use_lzo = FALSE;
	bkpinfo->use_gzip = TRUE;
	mr_asprintf(bkpinfo->zip_exe, "%s", "gzip");
	mr_asprintf(bkpinfo->zip_suffix, "%s", "gz");
}

read_cfg_var(cfg_file, "use-comp", value);
if (strstr(value, "yes")) {
	bkpinfo->use_lzma = FALSE;
	bkpinfo->use_lzo = FALSE;
	bkpinfo->use_gzip = FALSE;
	mr_asprintf(bkpinfo->zip_exe, "%s", "xz");
	mr_asprintf(bkpinfo->zip_suffix, "%s", "xz");
}

value[0] = '\0';
read_cfg_var(cfg_file, "differential", value);
if (!strcmp(value, "yes") || !strcmp(value, "1")) {
	bkpinfo->differential = TRUE;
}
log_msg(2, "differential var = '%s'", value);
if (bkpinfo->differential) {
	log_msg(2, "THIS IS A DIFFERENTIAL BACKUP");
} else {
	log_msg(2, "This is a regular (full) backup");
}

read_cfg_var(cfg_file, "please-dont-eject", value);
tmp1 = call_program_and_get_last_line_of_output("cat "CMDLINE);
if (value[0] || strstr(tmp1, "donteject")) {
	bkpinfo->please_dont_eject = TRUE;
	log_msg(2, "Ok, I shan't eject when restoring! Groovy.");
}
mr_free(tmp1);

/* TODO: Read here the boot_* variables */
read_cfg_var(cfg_file, "boot-type", value);
if (!strcmp(value, "BIOS")) {
	bkpinfo->boot_type = BIOS;
} else if (!strcmp(value, "UEFI")) {
	bkpinfo->boot_type = UEFI;
} else {
	log_msg(1, "No known boot type found");
}
log_msg(1, "Found %s boot type",value);

if (bkpinfo->backup_media_type == netfs) {
	if (!cfgf) {
		if (bkpinfo->netfs_mount) {
			log_msg(2, "netfs_mount remains %s", bkpinfo->netfs_mount);
		}
		if (bkpinfo->netfs_remote_dir != NULL) {
			log_msg(2, "netfs_remote_dir remains %s", bkpinfo->netfs_remote_dir);
		}
		log_msg(2, "...cos it wouldn't make sense to abandon the values that GOT ME to this config file in the first place");
	} else {
		read_cfg_var(cfg_file, "netfs-server-mount", value);
		mr_free(bkpinfo->netfs_mount);
		mr_asprintf(bkpinfo->netfs_mount, "%s", value);
		if (bkpinfo->netfs_mount != NULL) {
			log_msg(2, "netfs_mount is %s", bkpinfo->netfs_mount);
		}

		read_cfg_var(cfg_file, "iso-dir", value);
		mr_free(bkpinfo->netfs_remote_dir);
		mr_asprintf(bkpinfo->netfs_remote_dir, "%s", value);
		if (bkpinfo->netfs_remote_dir != NULL) {
			log_msg(2, "netfs_remote_dir is %s", bkpinfo->netfs_remote_dir);
		}

		if (bkpinfo->netfs_proto != NULL) {
			log_msg(2, "netfs_proto is %s", bkpinfo->netfs_proto);
		}
		if (bkpinfo->netfs_user != NULL) {
			log_msg(2, "netfs_user is %s", bkpinfo->netfs_user);
		}
	}
	tmp1 = call_program_and_get_last_line_of_output("cat "CMDLINE);
	if (strstr(tmp1, "pxe")) {
		/* We need to override values in PXE mode as it's
		* already done in start-netfs */
		envtmp1 = getenv("netfsmount");
		if (envtmp1 == NULL) {
			fatal_error("no netfsmount variable in environment");
		}
		envtmp2 = getenv("dirimg");
		if (envtmp2 == NULL) {
			fatal_error("no dirimg variable in environment");
		}
		mr_free(bkpinfo->netfs_mount);
		mr_asprintf(bkpinfo->netfs_mount, "%s", envtmp1);

		mr_free(bkpinfo->netfs_remote_dir);
		mr_asprintf(bkpinfo->netfs_remote_dir, "%s", envtmp2);
	}
	mr_free(tmp1);
} else if (bkpinfo->backup_media_type == iso) {
	/* Patch by Conor Daly 23-june-2004 
 	* to correctly mount iso-dev and set a sensible
 	* isodir in disaster recovery mode
 	*/
	mr_asprintf(old_isodir, "%s", bkpinfo->isodir);
	if (read_cfg_var(cfg_file, "iso-mnt", iso_mnt) != 0) {
		iso_mnt[0] = '\0';
	}
	if (read_cfg_var(cfg_file, "iso-dir", iso_path) != 0) {
		iso_path[0] = '\0';
	}
	mr_asprintf(bkpinfo->isodir, "%s%s", iso_mnt, iso_path);
	if (!strcmp(bkpinfo->isodir,"")) {
		mr_asprintf(bkpinfo->isodir, "%s", old_isodir);
	}

	if ((!bkpinfo->disaster_recovery) || (g_isodir_device[0] != '\0')) {
		if (strcmp(old_isodir, bkpinfo->isodir)) {
			log_it("user nominated isodir %s differs from archive, keeping user's choice: %s\n", bkpinfo->isodir, old_isodir );
			mr_asprintf(bkpinfo->isodir, "%s", old_isodir);
		}
	}
	mr_free(old_isodir);

	if (g_isodir_device[0] == '\0') {
		/*  We don't want to loose our conf made earlier in iso_fiddly_bits 
		 *  so go here only if job not done already */
		read_cfg_var(cfg_file, "iso-dev", g_isodir_device);
	}

	log_msg(2, "isodir=%s; iso-dev=%s", bkpinfo->isodir, g_isodir_device);

	if (bkpinfo->disaster_recovery) {
		if (is_this_device_mounted(g_isodir_device)) {
			log_msg(2, "NB: isodir is already mounted");
			/* Find out where it's mounted */
			mr_asprintf(command, "mount | grep -E '^%s' | tail -n1 | cut -d' ' -f3", g_isodir_device);
			log_it("command = %s", command);
			tmp1 = call_program_and_get_last_line_of_output(command);
			log_it("res of it = %s", tmp1);
			sprintf(iso_mnt, "%s", tmp1);
			mr_free(command);
			mr_free(tmp1);
		} else {
			sprintf(iso_mnt, "/tmp/isodir");
			mr_asprintf(tmp1, "mkdir -p %s", iso_mnt);
			run_program_and_log_output(tmp1, 5);
			mr_free(tmp1);

			mr_asprintf(tmp1, "mount %s %s", g_isodir_device, iso_mnt);
			if (run_program_and_log_output(tmp1, 3)) {
				log_msg(1, "Unable to mount isodir. Perhaps this is really an optical based backup?");
				bkpinfo->backup_media_type = optical;
				mr_asprintf(bkpinfo->media_device, "%s", "/dev/cdrom");	/* superfluous */
				mr_free(bkpinfo->isodir);
				iso_mnt[0] = iso_path[0] = '\0';
				if (mount_media(MNT_CDROM)) {
					mr_free(tmp1);
					fatal_error("Unable to mount isodir. Failed to mount CD-ROM as well.");
				} else {
					log_msg(1, "You backed up to disk, then burned some CDs.");
				}
			}
			mr_free(tmp1);
		}
		/* bkpinfo->isodir should now be the true path to prefix-1.iso etc... */
		/*  except when already done in iso_fiddly_bits */
		if ((bkpinfo->backup_media_type == iso) && (g_isodir_device[0] == '\0')) {
			mr_asprintf(bkpinfo->isodir, "%s%s", iso_mnt, iso_path);
		}
	}
}
mr_free(cfg_file);

if (media_specified_by_user != none) {
	if (g_restoring_live_from_cd) {
		if (bkpinfo->backup_media_type != media_specified_by_user) {
			log_msg(2, "bkpinfo->backup_media_type != media_specified_by_user, so I'd better ask :)");
			interactively_obtain_media_parameters_from_user(FALSE);
			media_specified_by_user = bkpinfo->backup_media_type;
			get_cfg_file_from_archive();
		}
	}
	bkpinfo->backup_media_type = media_specified_by_user;
}
g_backup_media_type = bkpinfo->backup_media_type;
return (0);

}

/**************************************************************************
*END_READ_CFG_FILE_INTO_BKPINFO                                          *
**************************************************************************/




/**
 * Allow the user to edit the filelist and biggielist.
 * The filelist is unlinked after it is read.
 * @param bkpinfo The backup information structure. Fields used:
 * - @c bkpinfo->backup_media_type
 * - @c bkpinfo->isodir
 * - @c bkpinfo->media_device
 * - @c bkpinfo->tmpdir
 * @return The filelist structure containing the information read from disk.
 */
struct
s_node *process_filelist_and_biggielist()
{
struct s_node *filelist;

char *command = NULL;
char *old_pwd = NULL;
char *q;
int res = 0;
pid_t pid;
bool extract_mountlist_stub = FALSE;

assert(bkpinfo != NULL);

/* If those files already exist, do not overwrite them later on */
if (does_file_exist("/"MOUNTLIST_FNAME_STUB)) {
	extract_mountlist_stub = FALSE;
} else {
	extract_mountlist_stub = TRUE;
}

if (does_file_exist(g_filelist_full)
&& does_file_exist(g_biggielist_txt)) {
	log_msg(1, "%s exists", g_filelist_full);
	log_msg(1, "%s exists", g_biggielist_txt);
	log_msg(2,
		"Filelist and biggielist already recovered from media. Yay!");
} else {
	old_pwd = mr_getcwd();
	if (chdir(bkpinfo->tmpdir)) {
		// FIXME
	}
	log_msg(1, "chdir(%s)", bkpinfo->tmpdir);
	log_to_screen("Extracting filelist and biggielist from media...");
	unlink("/tmp/filelist.full");
	unlink(FILELIST_FULL_STUB);
	if (IS_THIS_A_STREAMING_BACKUP(bkpinfo->backup_media_type)) {
		mr_asprintf(command, "tar -b %ld -zxf %s ./%s ./%s ./%s ./%s ./%s", bkpinfo->internal_tape_block_size, bkpinfo->media_device, MOUNTLIST_FNAME_STUB, BIGGIELIST_TXT_STUB, FILELIST_FULL_STUB, IWANTMYLVM_STUB, MONDO_CFG_FILE_STUB);
		log_msg(1, "tarcommand = %s", command);
		run_program_and_log_output(command, 1);
		mr_free(command);

		if (!does_file_exist(FILELIST_FULL_STUB)) {
			/* Doing that allow us to remain compatible with pre-2.2.5 versions */
			log_msg(2, "pre-2.2.4 compatible mode on");
			mr_asprintf(command, "tar -b %ld -zxf %s %s %s %s %s %s", bkpinfo->internal_tape_block_size, bkpinfo->media_device, MOUNTLIST_FNAME_STUB, BIGGIELIST_TXT_STUB, FILELIST_FULL_STUB, IWANTMYLVM_STUB, MONDO_CFG_FILE_STUB);
			log_msg(1, "tarcommand = %s", command);
			run_program_and_log_output(command, 1);
			mr_free(command);
		}
	} else {
		log_msg(2, "Calling insist_on_this_cd_number; bkpinfo->isodir=%s", bkpinfo->isodir);
		insist_on_this_cd_number(1);
		log_msg(2, "Back from iotcn");
		run_program_and_log_output("mount", 1);
		mr_asprintf(command, "tar -zxf %s/images/all.tar.gz ./%s ./%s ./%s ./%s ./%s", MNT_CDROM, MOUNTLIST_FNAME_STUB, BIGGIELIST_TXT_STUB, FILELIST_FULL_STUB, IWANTMYLVM_STUB, MONDO_CFG_FILE_STUB);

		log_msg(1, "tarcommand = %s", command);
		run_program_and_log_output(command, 1);
		mr_free(command);

		if (!does_file_exist(FILELIST_FULL_STUB)) {
			/* Doing that allow us to remain compatible with pre-2.2.5 versions */
			log_msg(2, "pre-2.2.4 compatible mode on");
			mr_asprintf(command, "tar -zxf %s/images/all.tar.gz %s %s %s %s %s", MNT_CDROM, MOUNTLIST_FNAME_STUB, BIGGIELIST_TXT_STUB, FILELIST_FULL_STUB, IWANTMYLVM_STUB, MONDO_CFG_FILE_STUB);

			log_msg(1, "tarcommand = %s", command);
			run_program_and_log_output(command, 1);
			mr_free(command);
		}
		if (!does_file_exist(BIGGIELIST_TXT_STUB)) {
			fatal_error("all.tar.gz did not include " BIGGIELIST_TXT_STUB);
		}
		if (!does_file_exist(FILELIST_FULL_STUB)) {
			fatal_error("all.tar.gz did not include " FILELIST_FULL_STUB);
		}
	}
	mr_asprintf(command, "cp -f %s %s", MONDO_CFG_FILE_STUB, g_mondo_cfg_file);
	run_program_and_log_output(command, FALSE);
	mr_free(command);

	mr_asprintf(command, "cp -f %s/%s %s", bkpinfo->tmpdir,	BIGGIELIST_TXT_STUB, g_biggielist_txt);
	log_msg(1, "command = %s", command);
	paranoid_system(command);
	mr_free(command);

	mr_asprintf(command, "ln -sf %s/%s %s", bkpinfo->tmpdir, FILELIST_FULL_STUB, g_filelist_full);
	log_msg(1, "command = %s", command);
	paranoid_system(command);
	mr_free(command);
	}

	if (am_I_in_disaster_recovery_mode()
	&&
	/* If it was there, do not overwrite it */
	(extract_mountlist_stub) 
	&&
	ask_me_yes_or_no("Do you want to retrieve the mountlist as well?"))
	{
		mr_asprintf(command, "ln -sf %s/%s /tmp", MOUNTLIST_FNAME_STUB,	bkpinfo->tmpdir);
		paranoid_system(command);
		mr_free(command);
	}

	if (chdir(old_pwd)) {
		// FIXME
	}
	mr_free(old_pwd);

	if (!does_file_exist(g_biggielist_txt)) {
		log_msg(1, "Warning - %s not found", g_biggielist_txt);
	}
	if (!does_file_exist(g_filelist_full)) {
		log_msg(1, "Warning - %s does not exist", g_filelist_full);
	}

	log_msg(2, "Forking");
	pid = fork();
	switch (pid) {
	case -1:
		fatal_error("Forking error");
		break;

	case 0:
		log_to_screen("Pre-processing filelist");
		if (!does_file_exist(g_biggielist_txt)) {
			mr_asprintf(command, "echo -n > %s", g_biggielist_txt);
			paranoid_system(command);
			mr_free(command);
		}
		mr_asprintf(command, "grep -E '^/dev/.*' %s > %s", g_biggielist_txt, g_filelist_imagedevs);
		paranoid_system(command);
		mr_free(command);
		exit(0);
		break;

	default:
		open_evalcall_form("Pre-processing filelist");
		while (!waitpid(pid, (int *) 0, WNOHANG)) {
			usleep(100000);
			update_evalcall_form(0);
		}
		break;
	}
	close_evalcall_form();

	log_msg(3, "loading filelist");
	filelist = load_filelist(g_filelist_full);
	log_msg(3, "deleting original filelist");
	unlink(g_filelist_full);
	if (g_text_mode) {
		q = NULL;
		while (q == NULL) {
			printf("Restore which directory? --> ");
			mr_getline(q, stdin);
		}
		toggle_path_selection(filelist, q, TRUE);
		if (strlen(q) == 0) {
			res = 1;
		} else {
			res = 0;
		}
		mr_free(q);
	} else {
		res = edit_filelist(filelist);
	}
	if (res) {
		log_msg(2, "User hit 'cancel'. Freeing filelist and aborting.");
		free_filelist(filelist);
		return (NULL);
	}
	ask_about_these_imagedevs(g_filelist_imagedevs, g_imagedevs_restthese);
	close_evalcall_form();

	// NB: It's not necessary to add g_biggielist_txt to the filelist.full
	// file. The filelist.full file already contains the filename of EVERY
	// file backed up - regular and biggie files.

	// However, we do want to make sure the imagedevs selected by the user
	// are flagged for restoring.
	if (length_of_file(g_imagedevs_restthese) > 2) {
		add_list_of_files_to_filelist(filelist, g_imagedevs_restthese,
									  TRUE);
	}

	return (filelist);
}

/**************************************************************************
 *END_ PROCESS_FILELIST_AND_BIGGIELIST                                    *
 **************************************************************************/




/**
 * Make a backup copy of <tt>path_root</tt>/<tt>filename</tt>.
 * The backup filename is the filename of the original with ".pristine" added.
 * @param path_root The place where the filesystem is mounted (e.g. MNT_RESTORING).
 * @param filename The filename (absolute path) within @p path_root.
 * @return 0 for success, nonzero for failure.
 */
int backup_crucial_file(char *path_root, char *filename)
{
	char *tmp = NULL;
	char *command = NULL;
	int res;

	assert(path_root != NULL);
	assert_string_is_neither_NULL_nor_zerolength(filename);

	mr_asprintf(tmp, "%s/%s", path_root, filename);
	mr_asprintf(command, "cp -f %s %s.pristine", tmp, tmp);
	mr_free(tmp);

	res = run_program_and_log_output(command, 5);
	mr_free(command);
	return (res);
}

void offer_to_make_initrd() {

if (bkpinfo->restore_mode != nuke) {
	if (ask_me_yes_or_no
		("You will now be able to re-generate your initrd.\nThis is especially useful if you changed of hardware configuration, cloned, made P2V, used multipath...\nDo you need to do it ?")) {
		log_msg(1,"Launching shell for manual initrd recreation");
		if (does_file_exist(MNT_RESTORING"/etc/arch-release")) {
			popup_and_OK("You'll now be chrooted under your future / partition.\nEdit /etc/mkinitcpio.conf if needed and rebuild your initrd with the kernel preset name e.g.\nmkinitcpio -p kernel26\nThen type exit to finish.\n");
		} else {
			popup_and_OK("You'll now be chrooted under your future / partition.\nGo under /boot and rebuild your init(rd|ramfs) with\nmkinitrd -f -v initrd-2.x.y.img 2.x.y e.g.\nor initramfs, dracut, ... Then type exit to finish.");
		}
		mvaddstr_and_log_it(g_currentY, 0, "Modifying initrd...");
		if (!g_text_mode) {
			newtSuspend();
		}
		paranoid_system("chroot " MNT_RESTORING);
		if (!g_text_mode) {
			newtResume();
		}
		mvaddstr_and_log_it(g_currentY++, 74, "Done.");
	} else {
		return;
	}
} else {
	log_to_screen("Non-interactive mode: no way to give you the keyboard so that you re-generate your initrd. Hope it's OK");
	log_msg(1,"Non-interactive mode: no way to give you the keyboard so that you re-generate your initrd. Hope it's OK");
}
}


/**
 * Attempt to find the user's editor.
 * @return The editor found ("vi" if none could be found).
 * @note The returned string points to an allocated storage that needs to be freed by caller
 */
char *find_my_editor(void) {

	char *output = NULL ;

	if ((output = find_home_of_exe("pico")) == NULL) {
		if ((output = find_home_of_exe("nano")) == NULL) {
			if ((output = find_home_of_exe("e3em")) == NULL) {
				if ((output = find_home_of_exe("e3vi")) == NULL) {
					if ((output = find_home_of_exe("vim")) == NULL) {
						output = find_home_of_exe("vi");
					}
				}
			}
		}
	}
	if (output == NULL) {
		log_msg(2, " (find_my_editor) --- warning - no editor found, even vi");
	}
	return (output);
}




/**
 * Install LILO on the user's boot drive (determined by /etc/lilo.conf).
 * @param offer_to_run_stablilo If TRUE, then offer to hack the user's fstab for them.
 * @return 0 for success, nonzero for failure.
 */
int run_lilo(bool offer_to_run_stablilo)
{
  /** malloc **/
	char *command = NULL;
	char *tmp = NULL;
	char *editor = NULL;

	int res;
	int done;
	bool run_lilo_M = FALSE;

	if (!run_program_and_log_output
		("grep \"boot.*=.*/dev/md\" " MNT_RESTORING "/etc/lilo.conf", 1)) {
		run_lilo_M = TRUE;
	}

	if (offer_to_run_stablilo
		&& ask_me_yes_or_no("Did you change the mountlist or cloned the system ?"))

		/* interactive mode */
	{
		mvaddstr_and_log_it(g_currentY,
							0,
							"Modifying fstab and lilo.conf, and running LILO...                             ");
		mr_asprintf(command, "mr-stablilo-me");
		res = run_program_and_log_output(command, 3);
		mr_free(command);

		if (res) {
			popup_and_OK
				("You will now edit fstab and lilo.conf, to make sure they match your new mountlist.");
			for (done = FALSE; !done;) {
				editor = find_my_editor();
				if (editor == NULL) {
					popup_and_OK("No editor found. You won't be able to edit conf files");
				} else {
					if (!g_text_mode) {
						newtSuspend();
					}
					mr_asprintf(tmp, "%s " MNT_RESTORING "/etc/fstab", editor);
					paranoid_system(tmp);
					mr_free(tmp);
	
					mr_asprintf(tmp, "%s " MNT_RESTORING "/etc/lilo.conf", editor);
					paranoid_system(tmp);
					mr_free(tmp);
					mr_free(editor);
	
					if (!g_text_mode) {
						newtResume();
					}
//              newtCls();
					if (ask_me_yes_or_no("Edit them again?")) {
						continue;
					}
				}
				mr_free(editor);

				res = run_program_and_log_output("chroot " MNT_RESTORING " lilo -L", 3);
				if (res) {
					res = run_program_and_log_output("chroot " MNT_RESTORING " lilo", 3);
				}
				if (res) {
					done = ask_me_yes_or_no("LILO failed. Re-edit system files?");
				} else {
					done = TRUE;
				}
			}
		} else {
			log_to_screen("lilo.conf and fstab were modified OK");
		}
	} else
		/* nuke mode */
	{
		mvaddstr_and_log_it(g_currentY,
							0,
							"Running LILO...                                                 ");
		res =
			run_program_and_log_output("chroot " MNT_RESTORING " lilo -L",
									   3);
		if (res) {
			res =
				run_program_and_log_output("chroot " MNT_RESTORING " lilo",
										   3);
		}
		if (res) {
			mvaddstr_and_log_it(g_currentY++, 74, "Failed.");
			log_to_screen
				("Failed to re-jig fstab and/or lilo. Edit/run manually, please.");
		} else {
			mvaddstr_and_log_it(g_currentY++, 74, "Done.");
		}
	}
	if (run_lilo_M) {
		run_program_and_log_output("chroot " MNT_RESTORING
								   " lilo -M /dev/hda", 3);
		run_program_and_log_output("chroot " MNT_RESTORING
								   " lilo -M /dev/sda", 3);
	}
	return (res);
}

/**************************************************************************
 *END_RUN_LILO                                                            *
 **************************************************************************/


/**
 * Install a raw MBR onto @p bd.
 * @param offer_to_hack_scripts If TRUE, then offer to hack the user's fstab for them.
 * @param bd The device to copy the stored MBR to.
 * @return 0 for success, nonzero for failure.
 */
int run_raw_mbr(bool offer_to_hack_scripts, char *bd)
{
  /** malloc **/
	char *command = NULL;
	char *boot_device = NULL;
	char *tmp = NULL;
	char *editor;
	int res;
	int done;

	malloc_string(boot_device);
	assert_string_is_neither_NULL_nor_zerolength(bd);

	strcpy(boot_device, bd);

	if (offer_to_hack_scripts
		&& ask_me_yes_or_no("Did you change the mountlist or cloned the system ?")) {
		/* interactive mode */
		mvaddstr_and_log_it(g_currentY, 0, "Modifying fstab and restoring MBR...                           ");
		for (done = FALSE; !done;) {
			popup_and_OK("You will now edit fstab");
			editor = find_my_editor();
			if (editor == NULL) {
				popup_and_OK("No editor found. You won't be able to edit conf files");
			} else {
				if (!g_text_mode) {
					newtSuspend();
				}
				mr_asprintf(tmp, "%s " MNT_RESTORING "/etc/fstab", editor);

				paranoid_system(tmp);
				mr_free(tmp);
				if (!g_text_mode) {
					newtResume();
				}
			}
			mr_free(editor);

			popup_and_get_string("Boot device", "Please confirm/enter the boot device. If in doubt, try /dev/hda", boot_device, MAX_STR_LEN / 4);
			mr_asprintf(command, "mr-stabraw-me %s", boot_device);
			res = run_program_and_log_output(command, 3);
			mr_free(command);

			if (res) {
				done = ask_me_yes_or_no("Modifications failed. Re-try?");
			} else {
				done = TRUE;
			}
		}
	} else {
		/* nuke mode */
		mr_asprintf(command, "mr-raw %s /"MOUNTLIST_FNAME_STUB, boot_device);
		log_msg(2, "run_raw_mbr() --- command='%s'", command);

		mvaddstr_and_log_it(g_currentY, 0, "Restoring MBR...                                               ");
		res = run_program_and_log_output(command, 3);
		mr_free(command);
	}
	if (res) {
		mvaddstr_and_log_it(g_currentY++, 74, "Failed.");
		log_to_screen("MBR+fstab processed w/error(s). See %s for more info.", MONDO_LOGFILE);
	} else {
		mvaddstr_and_log_it(g_currentY++, 74, "Done.");
	}
	paranoid_free(boot_device);
	return (res);
}

/**************************************************************************
 *END_RUN_RAW_MBR                                                         *
 **************************************************************************/


/**
 * Install GRUB on @p bd.
 * @param offer_to_run_stabgrub If TRUE, then offer to hack the user's fstab for them.
 * @param bd The boot device where GRUB is installed.
 * @return 0 for success, nonzero for failure.
 */
int run_grub(bool offer_to_run_stabgrub, char *bd)
{
  /** malloc **/
	char *command = NULL;
	char *boot_device = NULL;
	char *tmp = NULL;
	char *editor = NULL;

	bool done;
	int res = 0;		/*  FALSE */
	bool mntlistchg = FALSE;
	FILE *fin = NULL;

	malloc_string(boot_device);

	assert_string_is_neither_NULL_nor_zerolength(bd);
	strcpy(boot_device, bd);

	if (offer_to_run_stabgrub && ask_me_yes_or_no("Did you change the mountlist or cloned the system ?")) {
		/* interactive mode */
		mvaddstr_and_log_it(g_currentY, 0, "Modifying fstab, mtab, device.map and menu.lst/grub.cfg, and running GRUB...");
		/*  Did we changed the mountlist ? If yes, then force editing conf files */
		if ((fin = fopen(MONDO_MNTLISTCHG, "r")) != NULL) {
			mntlistchg = TRUE;
		}
		for (done = FALSE; !done;) {
			popup_and_get_string("Boot device", "Please confirm/enter the boot device. If in doubt, try /dev/sda", boot_device, MAX_STR_LEN / 4);
			/*  Only try to adapt grub here first if the mountlist wasn't changed before */
			if (! mntlistchg) {
				mr_asprintf(command, "mr-stabgrub-me %s", boot_device);
				res = run_program_and_log_output(command, 1);
				mr_free(command);
			}

			if ((res) || (mntlistchg)) {
				if (res) {
					popup_and_OK("GRUB installation failed. You will now edit fstab, mtab, device.map and menu.lst/grub.cfg in order to fix grub install");
				} else {
					popup_and_OK("The mountlist was changed. You will now edit fstab, mtab, device.map and menu.lst/grub.cfg in order to fix grub install");
				}
				editor = find_my_editor();
				if (editor == NULL) {
					popup_and_OK("No editor found. You won't be able to edit conf files");
				} else {
					if (!g_text_mode) {
						newtSuspend();
					}
					mr_asprintf(tmp, "chroot %s %s /etc/fstab", MNT_RESTORING, editor);
					paranoid_system(tmp);
					mr_free(tmp);
	
					mr_asprintf(tmp, "chroot %s %s /etc/mtab", MNT_RESTORING, editor);
					paranoid_system(tmp);
					mr_free(tmp);
		
					if (does_file_exist(MNT_RESTORING"/boot/grub/menu.lst")) {
						mr_asprintf(tmp, "chroot %s %s /boot/grub/menu.lst", MNT_RESTORING, editor);
					} else if (does_file_exist(MNT_RESTORING"/boot/grub/grub.cfg")) {
						mr_asprintf(tmp, "chroot %s %s /boot/grub/grub.cfg", MNT_RESTORING, editor);
					} else if (does_file_exist(MNT_RESTORING"/boot/grub2/grub.cfg")) {
						mr_asprintf(tmp, "chroot %s %s /boot/grub2/grub.cfg", MNT_RESTORING, editor);
					}
					paranoid_system(tmp);
					mr_free(tmp);
		
					if (does_file_exist(MNT_RESTORING"/boot/grub/device.map")) {
						mr_asprintf(tmp, "chroot %s %s /boot/grub/device.map", MNT_RESTORING, editor);
					} else if (does_file_exist(MNT_RESTORING"/boot/grub2/device.map")) {
						mr_asprintf(tmp, "chroot %s %s /boot/grub2/device.map", MNT_RESTORING, editor);
	            	}
					paranoid_system(tmp);
					mr_free(tmp);
					if (!g_text_mode) {
						newtResume();
					}
				}
				mr_free(editor);

				mr_asprintf(command, "mr-stabgrub-me %s", boot_device);
				res = run_program_and_log_output(command, 1);
				mr_free(command);

				if (res) {
					popup_and_OK("GRUB installation failed. Please fix the conf files so that a manual install using 'grub-install' or similar command works. You are now chroot()'ed to your restored system. Please type 'exit' when you are done.");
					newtSuspend();
					paranoid_system("chroot " MNT_RESTORING);
					newtResume();
				} else {
					popup_and_OK("GRUB is now installed correctly");
					done = TRUE;
				}
			} else {
				done = TRUE;
			}
		}
	} else {
		/* nuke mode */
		if (!run_program_and_log_output("which mr-grub", FALSE)) {
			log_msg(1, "Yay! mr-grub found...");
			mr_asprintf(command, "mr-grub %s /"MOUNTLIST_FNAME_STUB, boot_device);
			log_msg(1, "command = %s", command);
		} else {
			// Or grub2-install
			mr_asprintf(command, "chroot " MNT_RESTORING " grub-install %s", boot_device);
			log_msg(1, "WARNING - mr-grub not found; using grub-install");
		}
		mvaddstr_and_log_it(g_currentY, 0, "Running GRUB...                                                 ");
		log_it("%s",command);
		res = run_program_and_log_output(command, 1);
		mr_free(command);

		if (res) {
			popup_and_OK("Because of bugs in GRUB's own installer, GRUB was not installed properly. Please install the boot loader manually now, using this chroot()'ed shell prompt. Type 'exit' when you have finished.");
			newtSuspend();
			paranoid_system("chroot " MNT_RESTORING);
			newtResume();
		}
	}
	if (res) {
		mvaddstr_and_log_it(g_currentY++, 74, "Failed.");
		log_to_screen
			("GRUB ran w/error(s). See %s for more info.", MONDO_LOGFILE);
		log_msg(1, "Type:-");
		log_msg(1, "    mr-mount-me");
		log_msg(1, "    chroot " MNT_RESTORING);
		log_msg(1, "    mount /boot if needed");
		log_msg(1, "    grub-install '(hd0)'");
		log_msg(1, "    exit");
		log_msg(1, "    mr-unmount-me");
		log_msg(1,
				"If you're really stuck, please e-mail the mailing list.");
	} else {
		mvaddstr_and_log_it(g_currentY++, 74, "Done.");
	}
	paranoid_free(boot_device);

	return (res);
}

/**************************************************************************
 *END_RUN_GRUB                                                            *
 **************************************************************************/


/**
 * Install the user's boot loader in the MBR.
 * Currently LILO, GRUB, RAW (dd of MBR), and the FreeBSD bootloader are supported.
 * @param offer_to_hack_scripts If TRUE, then offer to hack the user's fstab for them.
 * @return 0 for success, nonzero for failure.
 */
int run_boot_loader(bool offer_to_hack_scripts)
{
	int res;
	int retval = 0;

  /** malloc *******/
	char *device = NULL;
	char *disk = NULL;
	char *name = NULL;
	char *type = NULL;
	char *cmd = NULL;

	malloc_string(device);
	malloc_string(name);
	malloc_string(type);

	/* In order to have a working bootloader, we need to have all devices
	 * ready in the chroot. If they are not there (udev) then copy them from
	 * the current /dev location 
	 */
	mr_asprintf(cmd,"tar cf - /dev | ( cd %s ; tar xf - )",MNT_RESTORING);
	run_program_and_log_output(cmd, 3);
	mr_free(cmd);

	backup_crucial_file(MNT_RESTORING, "/etc/fstab");
	backup_crucial_file(MNT_RESTORING, "/boot/grub/menu.lst");
	backup_crucial_file(MNT_RESTORING, "/boot/grub/grub.cfg");
	backup_crucial_file(MNT_RESTORING, "/boot/grub2/grub.cfg");
	backup_crucial_file(MNT_RESTORING, "/etc/lilo.conf");
	backup_crucial_file(MNT_RESTORING, "/boot/grub/device.map");
	backup_crucial_file(MNT_RESTORING, "/boot/grub2/device.map");
	backup_crucial_file(MNT_RESTORING, "/etc/mtab");
	read_cfg_var(g_mondo_cfg_file, "bootloader.device", device);
	read_cfg_var(g_mondo_cfg_file, "bootloader.name", name);
	read_cfg_var(g_mondo_cfg_file, "boot-type", type);
	log_msg(2, "run_boot_loader: device='%s', name='%s', type='%s'", device, name, type);
	sync();

	offer_to_make_initrd();

	disk = truncate_to_drive_name(device);
	if (strcmp(type,"BIOS") == 0) {
		// Force installation of a MBR bootloader brought by mindi on the disk
		// in case none was available and we then start from a partition
		log_msg(2, "Reinstalling mbr.bin on %s", disk);
		mr_system("dd bs=440 count=1 conv=notrunc if=/tmp/mbr.bin of=%s &> /dev/null",disk);
	} else {
		// Same for GPT
		log_msg(2, "Reinstalling gptmbr.bin on %s", disk);
		mr_system("dd bs=440 count=1 conv=notrunc if=/tmp/gptmbr.bin of=%s &> /dev/null",disk);
	}

	// Now reinstall bootloader
	if (!strcmp(name, "LILO")) {
		res = run_lilo(offer_to_hack_scripts);
	} else if (!strcmp(name, "GRUB")) {
		res = run_grub(offer_to_hack_scripts, device);
	} else if (!strcmp(name, "RAW")) {
		res = run_raw_mbr(offer_to_hack_scripts, device);
	}
#ifdef __FreeBSD__
	else if (!strcmp(name, "BOOT0")) {
		mr_asprintf(tmp, "boot0cfg -B %s", device);
		res = run_program_and_log_output(tmp, FALSE);
		paranoid_free(tmp);
	} else {
		mr_asprintf(tmp, "ls /dev | grep -Eq '^%ss[1-4].*'", device);
		if (!system(tmp)) {
			mr_free(tmp);
			mr_asprintf(tmp, MNT_RESTORING "/sbin/fdisk -B %s", device);
			res = run_program_and_log_output(tmp, 3);
		} else {
			log_msg(1, "I'm not running any boot loader. You have a DD boot drive. It's already loaded up.");
		}
		mr_free(tmp);
	}
#else
	else {
		log_to_screen
			("Unable to determine type of boot loader. Defaulting to LILO.");
		res = run_lilo(offer_to_hack_scripts);
	}
#endif
	retval += res;
	if (res) {
		log_to_screen("Your boot loader returned an error");
	} else {
		log_to_screen("Your boot loader ran OK");
	}
	paranoid_free(device);
	paranoid_free(name);
	return (retval);
}

/**************************************************************************
 *END_ RUN_BOOT_LOADER                                                    *
 **************************************************************************/



/**
 * malloc() and set sensible defaults for the mondorestore filename variables.
 * @param bkpinfo The backup information structure. Fields used:
 * - @c bkpinfo->tmpdir
 * - @c bkpinfo->disaster_recovery
 */
void setup_MR_global_filenames()
{
	assert(bkpinfo != NULL);

	malloc_string(g_biggielist_txt);
	malloc_string(g_filelist_full);
	malloc_string(g_filelist_imagedevs);
	malloc_string(g_imagedevs_restthese);
	malloc_string(g_mondo_cfg_file);
	malloc_string(g_mountlist_fname);
	malloc_string(g_mondo_home);
	malloc_string(g_isodir_device);
	malloc_string(g_isodir_format);

	sprintf(g_biggielist_txt, "%s/%s", bkpinfo->tmpdir, BIGGIELIST_TXT_STUB);
	sprintf(g_filelist_full, "%s/%s", bkpinfo->tmpdir, FILELIST_FULL_STUB);
	sprintf(g_filelist_imagedevs, "%s/tmp/filelist.imagedevs", bkpinfo->tmpdir);
//  sprintf(g_imagedevs_pot, "%s/tmp/imagedevs.pot", bkpinfo->tmpdir);
	sprintf(g_imagedevs_restthese, "%s/tmp/imagedevs.restore-these",
			bkpinfo->tmpdir);
	if (bkpinfo->disaster_recovery) {
		sprintf(g_mondo_cfg_file, "/%s", MONDO_CFG_FILE_STUB);
		sprintf(g_mountlist_fname, "/%s", MOUNTLIST_FNAME_STUB);
	} else {
		sprintf(g_mondo_cfg_file, "%s/%s", bkpinfo->tmpdir, MONDO_CFG_FILE_STUB);
		sprintf(g_mountlist_fname, "%s/%s", bkpinfo->tmpdir, MOUNTLIST_FNAME_STUB);
	}
}

/**************************************************************************
 *END_SET_GLOBAL_FILENAME                                                 *
 **************************************************************************/


/**
 * Copy @p input_file (containing the result of a compare) to @p output_file,
 * deleting spurious "changes" along the way.
 * @param output_file The output file to write with spurious changes removed.
 * @param input_file The input file, a list of changed files created by a compare.
 */
void streamline_changes_file(char *output_file, char *input_file)
{
	FILE *fin;
	FILE *fout;
	char *incoming = NULL;

	assert_string_is_neither_NULL_nor_zerolength(output_file);
	assert_string_is_neither_NULL_nor_zerolength(input_file);

	if (!(fin = fopen(input_file, "r"))) {
		log_OS_error(input_file);
		return;
	}
	if (!(fout = fopen(output_file, "w"))) {
		fatal_error("cannot open output_file");
	}
	for (mr_getline(incoming, fin); !feof(fin); mr_getline(incoming, fin)) {
		if (strncmp(incoming, "etc/adjtime", 11)
			&& strncmp(incoming, "etc/mtab", 8)
			&& strncmp(incoming, "tmp/", 4)
			&& strncmp(incoming, "boot/map", 8)
			&& !strstr(incoming, "incheckentry")
			&& strncmp(incoming, "etc/mail/statistics", 19)
			&& strncmp(incoming, "var/", 4))
			fprintf(fout, "%s", incoming);	/* don't need \n here, for some reason.. */
		mr_free(incoming);
	}
	mr_free(incoming);
	paranoid_fclose(fout);
	paranoid_fclose(fin);
}

/**************************************************************************
 *END_STREAMLINE_CHANGES_FILE                                             *
 **************************************************************************/


/**
 * Give the user twenty seconds to press Ctrl-Alt-Del before we nuke their drives.
 */
void twenty_seconds_til_yikes()
{
	int i;
	/* MALLOC * */
	char *tmp = NULL;

	if (does_file_exist("/tmp/NOPAUSE")) {
		return;
	}
	open_progress_form("CAUTION",
					   "Be advised: I am about to ERASE your hard disk(s)!",
					   "You may press Ctrl+Alt+Del to abort safely.",
					   "", 20);
	for (i = 0; i < 20; i++) {
		g_current_progress = i;
		mr_asprintf(tmp, "You have %d seconds left to abort.", 20 - i);
		update_progress_form(tmp);
		mr_free(tmp);
		sleep(1);
	}
	close_progress_form();
}

/**************************************************************************
 *END_TWENTY_SECONDS_TIL_YIKES                                            *
 **************************************************************************/


/**
 * Unmount all devices in @p p_external_copy_of_mountlist.
 * @param p_external_copy_of_mountlist The mountlist to guide the devices to unmount.
 * @return 0 for success, nonzero for failure.
 */
int unmount_all_devices(struct mountlist_itself
						*p_external_copy_of_mountlist)
{
	struct mountlist_itself *mountlist;
	int retval = 0, lino, res = 0, i;
	char *command = NULL;
	char *tmp = NULL;

	assert(p_external_copy_of_mountlist != NULL);

	mountlist = malloc(sizeof(struct mountlist_itself));
	memcpy((void *) mountlist, (void *) p_external_copy_of_mountlist, sizeof(struct mountlist_itself));
	sort_mountlist_by_mountpoint(mountlist, 0);

	run_program_and_log_output("df -m -P -T", 3);
	mvaddstr_and_log_it(g_currentY, 0, "Unmounting devices      ");
	open_progress_form("Unmounting devices", "Unmounting all devices that were mounted,", "in preparation for the post-restoration reboot.", "", mountlist->entries);
	if (chdir("/")) {
		// FIXME
	}
	for (i = 0; i < 10 && run_program_and_log_output("ps | grep buffer | grep -v \"grep buffer\"", TRUE) == 0; i++) {
		sleep(1);
		log_msg(2, "Waiting for buffer() to finish");
	}

	sync();

	/* after that what is logged is not copied on disk, typically the result of labelling */
	mr_asprintf(tmp, "cp -f %s " MNT_RESTORING "/var/log", MONDO_LOGFILE);
	if (run_program_and_log_output(tmp, FALSE)) {
		log_msg(1, "Error. Failed to copy log to the machine's /var/log dir. (Mounted read-only?)");
	}
	paranoid_free(tmp);
	if (does_file_exist("/tmp/DUMBASS-GENTOO")) {
		run_program_and_log_output("mkdir -p " MNT_RESTORING "/mnt/.boot.d", 5);
	}

	/* Unmounting the local /proc and /sys first */
	run_program_and_log_output("umount -d " MNT_RESTORING "/proc",3);
	run_program_and_log_output("umount -d " MNT_RESTORING "/sys",3);

	/*  Unmounting potential btrfs subvolumes */
	if (does_file_exist("/tmp/umount-btrfs-subvol")) {
		run_program_and_log_output("/tmp/umount-btrfs-subvol",3);
	}

	for (lino = mountlist->entries - 1; lino >= 0; lino--) {
		if (!strcmp(mountlist->el[lino].mountpoint, "lvm")) {
			continue;
		}
		mr_asprintf(tmp, "Unmounting device %s  ", mountlist->el[lino].device);
		update_progress_form(tmp);

		//TODO: this should be done in a certain order normally if there are cascading mount
		if (is_this_device_mounted(mountlist->el[lino].device)) {
			if (!strcmp(mountlist->el[lino].mountpoint, "swap")) {
				mr_asprintf(command, "swapoff %s", mountlist->el[lino].device);
			} else {
				if (!strcmp(mountlist->el[lino].mountpoint, "/1")) {
					mr_asprintf(command, "umount -d %s/", MNT_RESTORING);
					log_msg(3,
							"Well, I know a certain kitty-kitty who'll be sleeping with Mommy tonight...");
				} else {
					mr_asprintf(command, "umount -d " MNT_RESTORING "%s", mountlist->el[lino].mountpoint);

					/* To support latest Ubuntu where /var is a separate FS 
					 * Cf: http://linux.derkeiler.com/Mailing-Lists/Ubuntu/2007-04/msg01319.html 
					 * we need to create some dirs under the real / before unmounting it */
					if (!strcmp(mountlist->el[lino].mountpoint, "/")) {
						run_program_and_log_output("mkdir -p " MNT_RESTORING "/var/lock", FALSE);
						run_program_and_log_output("mkdir -p " MNT_RESTORING "/var/run", FALSE);
					}
				}
			}
			log_msg(10, "The 'umount' command is '%s'", command);
			res = run_program_and_log_output(command, 3);
			mr_free(command);
		} else {
			mr_strcat(tmp, "...not mounted anyway :-) OK");
			res = 0;
		}
		g_current_progress++;
		if (res) {
			mr_strcat(tmp, "...Failed");
			retval++;
			log_to_screen(tmp);
		} else {
			log_msg(2, tmp);
		}
		paranoid_free(tmp);
	}
	close_progress_form();
	if (retval) {
		mvaddstr_and_log_it(g_currentY++, 74, "Failed.");
	} else {
		mvaddstr_and_log_it(g_currentY++, 74, "Done.");
	}
	if (retval) {
		log_to_screen("Unable to unmount some of your partitions.");
	} else {
		log_to_screen("All partitions were unmounted OK.");
	}
	mr_asprintf(command, "mount");
	log_msg(4, "mount result after unmounting all FS", command);
	(void)run_program_and_log_output(command, 3);
	mr_free(command);
	free(mountlist);
	return (retval);
}

/**************************************************************************
 *END_UNMOUNT_ALL_DEVICES                                                 *
 **************************************************************************/
/* @} - end restoreUtilityGroup */

void wait_until_software_raids_are_prepped(int wait_for_percentage)
{
	struct raidlist_itself *raidlist;
	int unfinished_mdstat_devices = 9999;
	int i = 0;
	char *screen_message = NULL;

	raidlist = malloc(sizeof(struct raidlist_itself));

	assert(wait_for_percentage <= 100);
	log_it("wait_until_software_raids_are_prepped");
	while (unfinished_mdstat_devices > 0) {
	        // FIXME: Prefix '/dev/' should really be dynamic!
		if (parse_mdstat(MDSTAT_FILE, raidlist, "/dev/")) {
			log_to_screen("Sorry, cannot read %s", MDSTAT_FILE);
			log_msg(1,"Sorry, cannot read %s", MDSTAT_FILE);
			return;
		}
		for (unfinished_mdstat_devices = 0; i <= raidlist->entries; i++) {
			if (raidlist->el[i].progress < wait_for_percentage) {
				unfinished_mdstat_devices++;
				if (raidlist->el[i].progress == -1)	// delayed while another partition inits
				{
					continue;
				}
				log_msg(1,"Sync'ing %s (i=%d)", raidlist->el[i].raid_device, i);
				mr_asprintf(screen_message, "Sync'ing %s", raidlist->el[i].raid_device);
				open_evalcall_form(screen_message);
				mr_free(screen_message);

				while (raidlist->el[i].progress < wait_for_percentage) {
					log_msg(1,"Percentage sync'ed: %d", raidlist->el[i].progress);
					update_evalcall_form(raidlist->el[i].progress);
					sleep(2);
					// FIXME: Prefix '/dev/' should really be dynamic!
					if (parse_mdstat(MDSTAT_FILE, raidlist, "/dev/")) {
						break;
					}
				}
				close_evalcall_form();
			}
		}
	}
	paranoid_free(raidlist);
}
