/* my-stuff.h
   $Id: my-stuff.h 3879 2024-03-09 02:10:04Z bruno $
*/

#define HAVE_MALLOC 1

/**
 * @file
 * The main header file for Mondo.
 */

#if !defined(bool) && !defined(__cplusplus)
/**
 * Create the illusion of a Boolean type.
 */
#define bool int
#define TRUE 1
#define FALSE 0
#endif

#ifndef _MY_STUFF_H_
#define _MY_STUFF_H_

#ifndef __FreeBSD__
#include <getopt.h>
#endif
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/ioctl.h>
#include <sys/sem.h>
#include <sys/param.h>
#include <stdio.h>
#include <stdlib.h>
#ifndef  __USE_FILE_OFFSET64
#define  __USE_FILE_OFFSET64
#endif
#ifndef  __USE_LARGEFILE64
#define  __USE_LARGEFILE64
#endif
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <stddef.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <time.h>
#include <unistd.h>
#include <signal.h>
//#include <curses.h>
#include <newt.h>
#include <ctype.h>
#include <string.h>
#include <pthread.h>
#include <assert.h>

#if defined(DEBUG) && !__cplusplus
int count;
char trace_log[255];			/*buffer for logging */
char *trace_log_ptr;
#endif							/* DEBUG */

#define IA64_BOOT_SIZE "16384"	/* Should be coherent with mindi */
#define STD_PREFIX "mondorescue"	/* Should be coherent with mindi */

/**
 * The location where mindi media images are stored.
 */
#define MINDI_CACHE_REL "var/cache/mindi"	/* Should be coherent with mindi */
#define MINDI_CACHE "/"MINDI_CACHE_REL
#define MINDI_LOGFILE "/var/log/mindi.log"
#define MINDI_RUNFILE "/var/log/mindi.run"

/**
 * The location where mondo changed info are stored.
 */
#define MONDO_CACHE "/var/cache/mondo"
#define ARCHIVES_PATH MNT_CDROM"/archives"
#define BIGGIELIST ARCHIVES_PATH"/biggielist.txt"

/**
 * The RAID kernel proc file
 */
#define MDSTAT_FILE "/proc/mdstat"

#ifdef __FreeBSD__
#define CMDLINE "/tmp/cmdline"
#else
// Default Linux
#define CMDLINE "/proc/cmdline"
#endif

#undef assert

extern void _mondo_assert_fail(const char *file, const char *function,
							   int line, const char *exp);

/**
 * An assert macro that calls _mondo_assert_fail() when it fails.
 */
#ifdef NDEBUG
#define assert(exp) ((void)0)
#else
#define assert(exp) ((exp)?((void)0):_mondo_assert_fail(__FILE__, __FUNCTION__, __LINE__, #exp))
#endif

// Extra info for ACLs and SELINUX users
#define STAR_ACL_SZ "-xattr -acl"

#define CRC_M16	0xA001			///< Mask for crc16.
#define	CRC_MTT	0x1021			///< Mask for crc-ccitt.

#define FALSE 0					///< The ubiquitous FALSE macro.
#define TRUE 1					///< The even more ubiquitous TRUE macro.

#define SCREEN_LENGTH 25		///< The default size of the screen.
#define NOOF_ERR_LINES 6		///< The number of lines of log output to keep at the bottom of the screen.
#define ARBITRARY_MAXIMUM 2000	///< The maximum number of items showing at once in the mountlist or filelist editor.
#define MAX_TAPECATALOG_ENTRIES 16384	///< The maximum number of entries in the tape catalog.
#define MAX_MOUNTLIST_ENTRIES 8192	///< The maximum number of entries in the mountlist.
#define MAX_STR_LEN 512			///< The maximum length of almost all @p char buffers in Mondo.
                                        ///  Note: Make this divisible by eight to avoid aligment issues
                                        ///        on 64bit platforms like ia64.
#define MAXIMUM_RAID_DEVS 32	///< The maximum number of RAID devices in the raidlist.
#define MAXIMUM_ADDITIONAL_RAID_VARS 32	///< The maximum number of additional RAID variables per RAID device in the raidlist.
#define MAXIMUM_DISKS_PER_RAID_DEV 1024	///< The maximum number of disks per RAID device in the raidtab.

#define RAIDTAB_FNAME "/etc/raidtab"	///< The filename of the raidtab file, at least on Linux.

#define BLK_START_OF_BACKUP     1	///< Marker block: start a backup.
#define BLK_START_OF_TAPE       2	///< Marker block: start a tape.
#define BLK_START_AFIOBALLS	10	///< Marker block: start the afioball section.
#define BLK_STOP_AFIOBALLS	19	///< Marker block: stop the afioball section.
#define BLK_START_AN_AFIO_OR_SLICE    	20	///< Marker block: start an afioball or a slice.
#define BLK_STOP_AN_AFIO_OR_SLICE	29	///< Marker block: stop an afioball or a slice.
#define BLK_START_BIGGIEFILES	30	///< Marker block: start the biggiefile section.
#define BLK_STOP_BIGGIEFILES	39	///< Marker block: stop the biggiefile section.
#define BLK_START_A_NORMBIGGIE	40	///< Marker block: start a normal biggiefile.
#define BLK_START_A_PIHBIGGIE	41	///< Marker block: start a ntfsprog'd biggiefile
#define BLK_START_EXTENDED_ATTRIBUTES 45	///< Marker block: start xattr/acl info
#define BLK_STOP_EXTENDED_ATTRIBUTES 46	///< Marker block: stop xattr/acl info
#define BLK_START_EXAT_FILE     47
#define BLK_STOP_EXAT_FILE      48
#define BLK_STOP_A_BIGGIE	59	///< Marker block: stop a biggiefile.
#define BLK_START_FILE          80	///< Marker block: start a file (non-afio or slice).
#define BLK_STOP_FILE           89	///< Marker block: stop a file (non-afio or slice).
#define BLK_END_OF_TAPE         100	///< Marker block: end of tape.
#define BLK_END_OF_BACKUP       101	///< Marker block: end of backup.
#define BLK_ABORTED_BACKUP      102	///< Marker block: backup was aborted.

/// The external tape blocksize.
#ifdef EXTTAPE
#define TAPE_BLOCK_SIZE (long)EXTTAPE
#else
#define TAPE_BLOCK_SIZE 131072L	/* was 8192; 06/2002-->65536; 11/2002-->131072 */
#endif

#define DEFAULT_INTERNAL_TAPE_BLOCK_SIZE 32768	// Nov 2003?




#define SLICE_SIZE 4096			///< The size of a slice of a biggiefile.






/**
 * Determine whether @p x (t_bkptype) is a streaming backup.
 */
#define IS_THIS_A_STREAMING_BACKUP(x) (x == tape || x == udev || x == cdstream)


#define MONDO_MKISOFS_CMD "mkisofs"
#define MONDO_MKISOFS_OPT ""
#define MONDO_WODIM_OPT ""
#define MONDO_GENISOIMAGE_OPT ""
#define MONDO_XORRISO_OPT "-as mkisofs"
#define MONDO_MKISOFS " -J -r -v -p MondoRescue -publisher http://www.mondorescue.org -A MondoRescue -V _CD#_ "
#define MONDO_MKISOFS_CMS " -b syslinux/isolinux.bin -c syslinux/boot.cat -no-emul-boot -boot-load-size 4 -boot-info-table "
#define MONDO_UEFI_PREFIX " -b EFI/isolinux.bin -c EFI/boot.cat -no-emul-boot -boot-load-size 4 -boot-info-table -eltorito-alt-boot "
/* No space before this one as we may want to add another '-' before it */
#define MONDO_MKISOFS_UEFI "-efi-boot images/mindi-bootroot.img -no-emul-boot "

/**
 * @c mkisofs command to generate a bootable CD using isolinux, except for -o option and the directory to image.
 */
#define MONDO_MKISOFS_REGULAR_SYSLINUX	MONDO_MKISOFS

/**
 * @c mkisofs command to generate a bootable CD using LILO, except for -o option and the directory to image.
 */
#define MONDO_MKISOFS_REGULAR_LILO      MONDO_MKISOFS_REGULAR_SYSLINUX

/**
 * @c mkisofs command to generate a bootable CD using ELILO, except for -o option and the directory to image.
 */
#define MONDO_MKISOFS_REGULAR_ELILO      MONDO_MKISOFS"-no-emul-boot -b images/mindi-bootroot.img -c images/boot.cat "

/**
 * The stub name of the temporary ISO image to create, burn, and remove.
 */
#define MONDO_TMPISOS "temporary.iso"

/**
 * @c growisofs command to generate a bootable DVD using isolinux, except for the directory to image.
 */
#define MONDO_GROWISOFS_REGULAR_SYSLINUX "growisofs -use-the-force-luke "MONDO_MKISOFS_REGULAR_SYSLINUX

/**
 * @c growisofs command to generate a bootable DVD using LILO, except for the directory to image.
	 */
#define MONDO_GROWISOFS_REGULAR_ELILO     "growisofs -use-the-force-luke "MONDO_MKISOFS_REGULAR_ELILO

/**
 * @c growisofs command to generate a bootable DVD using LILO, except for the directory to image.
	 */
#define MONDO_GROWISOFS_REGULAR_LILO     "growisofs "MONDO_MKISOFS_REGULAR_SYSLINUX

/**
 * @c growisofs command to generate a nonbootable DVD, except for the directory to image.
 */
#define MONDO_GROWISOFS_NONBOOT          "growisofs -use-the-force-luke "MONDO_MKISOFS

/**
 * Welcome string displayed at the top of the newt interface.
 */
#define WELCOME_STRING "W E L C O M E   T O   M O N D O   R E S C U E"

/**
 * The maximum length of a filename in the tape catalog.
 */
#define MAX_TAPECAT_FNAME_LEN 32

/**
 * Assert that (@p x != NULL) and (@p x[0] != '\\0').
 */
#define assert_string_is_neither_NULL_nor_zerolength(x) {assert(x!=NULL);assert(x[0]!='\0');}

/**
 * Log the file, line, Mondo error message, and OS error message (errno).
 */
#define log_OS_error(x) {log_msg(0, "%s (%s)", x, strerror(errno));}

/**
 * Assert that (@p x != NULL).
 */
#define assert_pointer_is_not_NULL(x) {assert(x!=NULL);}

/**
 * close() @p x and log a message if it fails.
 */
#define paranoid_close(x) {if(close(x)) {log_msg(5, "close err");} x=-999; }

/**
 * fclose() @p x and log a message if it fails.
 */
#define paranoid_fclose(x) {if(fclose(x)) {log_msg(5, "fclose err");} x=NULL; }

/**
 * pclose() @p x and log a message if it fails.
 */
#define paranoid_pclose(x) {if(pclose(x)) {log_msg(5, "pclose err");} x=NULL; }

/**
 * Run the command @p x and log it if it fails.
 */
#define paranoid_system(x) {if(system(x)) log_msg(4, x); }

/**
 * Free @p x and set it to NULL.
 */
#define paranoid_free(x) {if (x != NULL) free(x); x=NULL;}

/**
 * Free variables and call finish(@p x).
 */
#define paranoid_MR_finish(x) {free_MR_global_filenames (); finish(x); }

/**
 * Yes, we want malloc() to help us fix bugs.
 */
#define MALLOC_CHECK_ 1

/**
 * Malloc @p x to be MAX_STR_LEN bytes and call fatal_error() if we're out of memory.
 */
#define malloc_string(x) { x = (char *)malloc(MAX_STR_LEN); if (!x) { fatal_error("Unable to malloc"); } x[0] = x[1] = '\0'; }

/**
 * Path to the location the hard drive is mounted on during a restore.
 */
#define MNT_RESTORING "/mnt/RESTORING"

/** @def VANILLA_SCSI_CDROM The first SCSI CD-ROM in the system (most likely to be the one to write to). */
/** @def VANILLA_SCSI_TAPE  The first SCSI tape in the system (most likely to be the one towrite to. */
/** @def DONT_KNOW_HOW_TO_EVALUATE_THIS_DEVICE_TYPE A string whose presence in a device name indicates the
 * inability to check this device for errors in the mountlist. */
/** @def RAID_DEVICE_STUB The stub of a RAID device (set up RAID if we find it). */
/** @def ALT_TAPE The first IDE tape in the system. */
/** @def MKE2FS_OR_NEWFS @c mke2fs or @c newfs, depending on the OS. */
/** @def CP_BIN The GNU @c cp binary to use. */
#ifdef __FreeBSD__
#define VANILLA_SCSI_CDROM	"/dev/cd0"
#define ALT_CDROM	"/dev/ad0"
#define VANILLA_SCSI_TAPE	"/dev/sa0"
#define VANILLA_USB_DEVICE	"tobegivenbybsdguru"
#define DONT_KNOW_HOW_TO_EVALUATE_THIS_DEVICE_TYPE	"/dev/vinum/"
#define RAID_DEVICE_STUB	DONT_KNOW_HOW_TO_EVALUATE_THIS_DEVICE_TYPE
#define ALT_TAPE		"/dev/ast0"
#define MKE2FS_OR_NEWFS	"newfs"
#define CP_BIN		"gcp"
#else
#define VANILLA_SCSI_CDROM	"/dev/sr0"
#define ALT_CDROM	"/dev/scd0"
#define VANILLA_SCSI_TAPE	"/dev/st0"
#define VANILLA_USB_DEVICE	"/dev/hda"
#define DONT_KNOW_HOW_TO_EVALUATE_THIS_DEVICE_TYPE	"/dev/md"
#define RAID_DEVICE_STUB	DONT_KNOW_HOW_TO_EVALUATE_THIS_DEVICE_TYPE
#define ALT_TAPE		"/dev/ht0"
#define MKE2FS_OR_NEWFS	"mke2fs"
#define CP_BIN		"cp"
#endif


/**
 * The template for a filelist filename.
 * The first argument (%s) is the tempdir and the second (%d) is the filelist number.
 */
#define FILELIST_FNAME_RAW_SZ "%s/filelist.%ld"

#define XATTR_LIST_FNAME_RAW_SZ      "%s/xattr_list.%ld.gz"
#define XATTR_BIGGLST_FNAME_RAW_SZ   "%s/xattr_list.big.gz"
#define ACL_LIST_FNAME_RAW_SZ        "%s/acl_list.%ld.gz"
#define ACL_BIGGLST_FNAME_RAW_SZ     "%s/acl_list.big.gz"

#define BIGGIEMAXSIZE			64L*1024L
/**
 * The template for an afioball filename.
 * The first argument (%s) is the tempdir and the second (%d) is the filelist number.
 */
#define AFIOBALL_FNAME_RAW_SZ (bkpinfo->use_star)?"%s/tmpfs/%ld.star.%s":"%s/tmpfs/%ld.afio.%s"
#define ARCH_THREADS 2			///< The number of simultaneous threads running afio in the background.
#define ARCH_BUFFER_NUM (ARCH_THREADS*4)	// Number of permissible queued afio files
#define FORTY_SPACES "                                         "	///< 40 spaces.
#define DO_MBR_PLEASE "/tmp/DO-MBR-PLEASE"
#define MONDO_MNTLISTCHG "/tmp/mountlist.changed"

extern void (*log_debug_msg) (int debug_level, const char *szFile,
							  const char *szFunction, int nLine,
							  const char *fmt, ...);
extern void standard_log_debug_msg(int debug_level, const char *szFile,
								   const char *szFunction, int nLine,
								   const char *fmt, ...);

/**
 * Compatibility define to change log_it() calls to log_debug_msg() calls.
 */
#define log_it(format, args...) log_debug_msg(2, __FILE__, __FUNCTION__, __LINE__, format, ## args)

/**
 * Macro to log a message along with file, line, and function information.
 */
#define log_msg(level, format, args...) log_debug_msg(level, __FILE__, __FUNCTION__, __LINE__, format, ## args)

#define DEFAULT_DVD_DISK_SIZE 4480	///< The default size (in MB) of a DVD disk, unless the user says otherwise.

#define DEFAULT_DEBUG_LEVEL 4	///< By default, don't log messages with a loglevel higher than this.

#define SZ_NTFSPROG_VOLSIZE "1048576"	// was 4096
#define NTFSPROG_PARAMS "-z0 -V" SZ_NTFSPROG_VOLSIZE " -o -b -d -g1"

#define MNT_CDROM "/mnt/cdrom"

#define DEFAULT_MR_LOGLEVEL 4

#endif							/* _MY_STUFF_H_ */
