/***************************************************************************
$Id: libmondo-cli.c 3843 2024-03-06 01:07:54Z bruno $
*******************************************************************/

/**
 * @file
 * Functions for handling command-line arguments passed to mondoarchive.
 */

/** @def BOOT_LOADER_CHARS The characters allowed for boot loader on this platform. */

#include <pthread.h>
#include "my-stuff.h"
#include "mr_mem.h"
#include "mr_str.h"
#include "mondostructures.h"
#include "libmondo-cli-EXT.h"
#include "libmondo.h"

extern int g_loglevel;
extern bool g_text_mode;
extern bool g_fail_immediately;
extern char g_startdir[MAX_STR_LEN];	///< ????? @bug ?????
extern char *MONDO_OPTIONS;

/*@ file pointer **************************************************/
extern FILE *g_tape_stream;

/*@ long long *****************************************************/
extern long long g_tape_posK;

/*@ long **********************************************************/
extern long g_noof_sets;

/*@ bool******** **************************************************/
bool g_debugging = FALSE;		///< ????? @bug ????? @ingroup globalGroup
bool g_running_live = FALSE;	///< ????? @bug ????? @ingroup globalGroup
extern bool g_cd_recovery;

extern void setup_tmpdir(char *path);
extern void setup_scratchdir(char *path);
void mr_make_devlist_from_pathlist(char *pathlist, char mode);
extern double g_kernel_version;
extern int g_current_media_number;
extern pid_t g_main_pid;
extern char *resolve_softlinks_to_get_to_actual_device_file(char *);
extern char *mr_popup_and_get_string(char *, char *, char *);
extern char *call_program_and_get_last_line_of_output(const char *);

extern t_boot mr_boot_type(void);

/* Do we use extended attributes and acl ? 
 * By default no, use -z option to force their usage */
extern char *g_getfacl;
extern char *g_getfattr;

/* Reference to global bkpinfo */
extern struct s_bkpinfo *bkpinfo;

extern void free_MR_global_filenames(void);

long g_max_biggie_size = BIGGIEMAXSIZE;

/**
 * @addtogroup cliGroup
 * @{
 */
/**
 * Populate @p bkpinfo from the command-line parameters stored in @p argc and @p argv.
 * @param argc The argument count, including the program name; @p argc passed to main().
 * @param argv The argument vector; @p argv passed to main().
 * @param bkpinfo The backup information structure to populate.
 * @return The number of problems with the command line (0 for success).
 */
int
handle_incoming_parameters(int argc, char *argv[])
{
	/*@ int *** */
	int res = 0;
	int retval = 0;
	int i = 0;

	/*@ buffers *************** */
	char flag_val[128][MAX_STR_LEN];
	bool flag_set[128];

	for (i = 0; i < 128; i++) {
		flag_val[i][0] = '\0';
		flag_set[i] = FALSE;
	}
	bkpinfo->media_size = 650; /* default */
	res = retrieve_switches_from_command_line(argc, argv, flag_val, flag_set);
	retval += res;
	if (!retval) {
		res = process_switches(flag_val, flag_set);
		retval += res;
	}

	log_msg(3, "Switches:-");
	for (i = 0; i < 128; i++) {
		if (flag_set[i]) {
			log_msg(3, "-%c %s", i, flag_val[i]);
		}
	}
	bkpinfo->boot_type = mr_boot_type();

	return (retval);
}


/**
 * Store the sizespec(s) stored in @p value into @p bkpinfo.
 * @param bkpinfo The backup information structure; the @c bkpinfo->media_size field will be populated.
 * @param value The sizespec (e.g. "2g", "40m").
 * @return 0, always.
 * @bug Return code not needed.
 */
int process_the_s_switch(char *value)
{
	assert(bkpinfo != NULL);
	assert(value != NULL);

	bkpinfo->media_size = -1;	/* dummy value */
	bkpinfo->media_size = friendly_sizestr_to_sizelong(value);
	log_msg(3, "media_size = %ld", bkpinfo->media_size);
	if (bkpinfo->media_size <= 0) {
		log_msg(1, "You gave media an invalid size %s", value);
		return (-1);
	}
	return (0);
}

/**
 * Process mondoarchive's command-line switches.
 * @param bkpinfo The backup information structure to populate.
 * @param flag_val An array of the argument passed to each switch (the letter is the index).
 * If a switch is not set or has no argument, the field in @p flag_val doesn't matter.
 * @param flag_set An array of <tt>bool</tt>s indexed by switch letter: TRUE if it's set,
 * FALSE if it's not.
 * @return The number of problems with the switches, or 0 for success.
 * @bug Maybe include a list of all switches (inc. intentionally undocumented ones not in the manual!) here?
 */
int
process_switches(char flag_val[128][MAX_STR_LEN], bool flag_set[128])
{

	/*@ ints *** */
	int i = 0;
	int retval = 0;

	/*@ buffers ** */
	char *tmp = NULL;
	char *tmp1 = NULL;
	char *tmp2 = NULL;
	char *psz = NULL;
	char *p = NULL;
	char *q = NULL;
	char *q2 = NULL;

	long itbs = 0L;

	struct stat buf;

	malloc_string(tmp);

	assert(bkpinfo != NULL);
	assert(flag_val != NULL);
	assert(flag_set != NULL);

	bkpinfo->internal_tape_block_size = DEFAULT_INTERNAL_TAPE_BLOCK_SIZE;

	/* compulsory */
	i = flag_set['c'] + flag_set['i'] + flag_set['n'] +
		flag_set['t'] + flag_set['u'] + flag_set['r'] +
		flag_set['w'] + flag_set['C'] + flag_set['U'];
	if ((i == 0) && (! bkpinfo->restore_data)) {
		retval++;
		log_to_screen("You must specify the media type\n");
	}
	if (i > 1) {
		retval++;
		log_to_screen("Please specify only one media type\n");
	}

	if (flag_set['K']) {
		g_loglevel = atoi(flag_val['K']);
         	log_msg(1,"Loglevel forced to %d",g_loglevel);
		if (g_loglevel < 3) {
			g_loglevel = 3;
		}
	}

	if ((flag_set['L'] && flag_set['0']) && (! bkpinfo->restore_data)) {
		retval++;
		log_to_screen("You cannot have 'no compression' _and_ LZOP.\n");
	}
	if (! bkpinfo->restore_data) {
		bkpinfo->backup_data = flag_set['O'];
	}
	bkpinfo->verify_data = flag_set['V'];

	if (flag_set['I'] && !bkpinfo->backup_data) {
		log_to_screen("-I switch is ignored if just verifying");
	}
	if (flag_set['E'] && !bkpinfo->backup_data) {
		log_to_screen("-E switch is ignored if just verifying");
	}

	if (!find_home_of_exe("afio")) {
		if (find_home_of_exe("star")) {
			flag_set['R'] = TRUE;
			log_msg(1, "Using star instead of afio");
		} else {
			fatal_error("Neither afio nor star is installed. Please install at least one.");
		}
	}

	if (flag_set['R']) {
		bkpinfo->use_star = TRUE;
		if (flag_set['L']) {
			fatal_error("You may not use star and lzop at the same time.");
		}
		if (!find_home_of_exe("star")) {
			fatal_error("Please install 'star' RPM or tarball if you are going to use -R. Thanks.");
		}
	}

	if ((flag_set['W']) && (! bkpinfo->restore_data)) {
		bkpinfo->nonbootable_backup = TRUE;
		log_to_screen("Warning - you have opted for non-bootable backup");
		if (flag_set['f'] || flag_set['l']) {
			log_to_screen
				("You don't need to specify bootloader or bootdevice");
		}
	}

	if (flag_set['I']) {
		if (flag_val['I'][0] == '-') {
			retval++;
			log_to_screen("Please supply a sensible value with '-I'\n");
		}
		if (!strcmp(flag_val['I'],"/")) {
			log_msg(2, "'/' is pleonastic.");
		}
		if (bkpinfo->include_paths && bkpinfo->include_paths[0]) {
			mr_strcat(bkpinfo->include_paths, "|");
		}

		mr_asprintf(tmp1, "%s", flag_val['I']);
		p = tmp1;
		q = tmp1;

		/* Cut the flag_val['I'] in parts containing all paths to test them */
		while (p != NULL) {
			q = strchr(p, '|');
			if (q != NULL) {
				*q = '\0';
				if ((stat(p, &buf) != 0) && (! bkpinfo->restore_data)) {
					log_msg(1, "ERROR ! %s doesn't exist", p);
					fatal_error("ERROR ! You specified a directory to include which doesn't exist");
				}
				p = q+1 ;
			} else {
				if ((stat(p, &buf) != 0) && (! bkpinfo->restore_data)) {
					log_msg(1, "ERROR ! %s doesn't exist", p);
					fatal_error("ERROR ! You specified a directory to include which doesn't exist");
				}
				p = NULL;
			}
		}
		mr_free(tmp1);
		mr_make_devlist_from_pathlist(flag_val['I'], 'I');
		log_msg(4, "Finished with the -I option");
	}

	if (g_kernel_version >= 2.6 && !flag_set['d'] && (flag_set['c'] || flag_set['w']) && (! bkpinfo->restore_data)) {
		fatal_error("If you are using the 2.6.x kernel, please specify the CD-R(W) device.");
	}


	if (flag_set['J']) {
		if (flag_set['I']) {
			retval++;
			log_to_screen("Please do not use -J in combination with -I. If you want to make a list of files to backup, that's fine, use -J <filename> but please don't muddy the waters by combining -J with -I. Thanks. :-)");
		}
		bkpinfo->make_filelist = FALSE;
		mr_asprintf(bkpinfo->include_paths, "%s", flag_val['J']);
	}

	if ((flag_set['c'] || flag_set['w'] || flag_set['C'] || flag_set['r']) && (! bkpinfo->restore_data)) {
		if (!flag_set['r'] && g_kernel_version <= 2.5 && strstr(flag_val['d'], "/dev/")) {
			fatal_error("Please don't give a /dev entry. Give a SCSI node for the parameter of the -d flag.");
		}
		if (flag_set['r'] && g_kernel_version <= 2.5 && !strstr(flag_val['d'], "/dev/")) {
			fatal_error("Please give a /dev entry, not a SCSI node, as the parameter of the -d flag.");
		}
		if (g_kernel_version >= 2.6 && !strstr(flag_val['d'], "/dev/")) {
			log_to_screen("Linus says 2.6 has a broken ide-scsi module. Proceed at your own risk...");
		}

		if (system("which cdrecord > /dev/null 2> /dev/null") && system("which dvdrecord > /dev/null 2> /dev/null")) {
			fatal_error("Please install dvdrecord/cdrecord and try again.");
		}
		if (flag_set['C']) {
			bkpinfo->cdrw_speed = atoi(flag_val['C']);
			if (bkpinfo->cdrw_speed < 1) {
				fatal_error("You specified a silly speed for a CD-R[W] drive");
			}
			if (!flag_set['L']) {
				log_to_screen("You must use -L with -C. Therefore I am setting it for you.");
				flag_set['L'] = 1;
				flag_val['L'][0] = '\0';
			}
		} else {
			log_msg(3, "flag_val['c'] = %s", flag_val['c']);
			log_msg(3, "flag_val['w'] = %s", flag_val['w']);
			if (flag_set['c']) {
				bkpinfo->cdrw_speed = atoi(flag_val['c']);
			} else if (flag_set['w']) {
				bkpinfo->cdrw_speed = atoi(flag_val['w']);
			} else if (flag_set['r']) {
				bkpinfo->cdrw_speed = 1;	/*atoi(flag_val['r']); */
			}

			if (bkpinfo->cdrw_speed < 1) {
				fatal_error("You specified a silly speed for a CD-R[W] drive");
			}
		}
	}

	if ((flag_set['t'] && !flag_set['d']) && (! bkpinfo->restore_data)) {
		log_it("Hmm! No tape drive specified. Let's see what we can do.");
		if (find_tape_device_and_size(flag_val['d'], tmp)) {
			fatal_error("Tape device not specified. I couldn't find it either.");
		}
		flag_set['d'] = TRUE;
		log_to_screen("You didn't specify a tape streamer device. I'm assuming %s", flag_val['d']);
	}

	if (flag_set['U'])			// USB
	{
		if (! flag_set['d']) {
			fatal_error("You need to specify a device file with -d for bootable USB device usage");
		}
		if ((!flag_set['s']) && (! bkpinfo->restore_data)) {
			fatal_error("You did not specify a size (-s) for your USB device. Aborting");
		}
	}

	if (flag_set['r'])			// DVD
	{
		if (flag_set['m']) {
			fatal_error("Manual CD tray (-m) not yet supported in conjunction w/ DVD drives. Drop -m.");
		}
		if (!flag_set['d']) {
			if (!find_dvd_device(flag_val['d'])) {
				flag_set['d'] = TRUE;
				log_to_screen("I guess DVD drive is at %s", flag_val['d']);
			}
		}
		if (strchr(flag_val['d'], ',')) {
			fatal_error("Please don't give a SCSI node. Give a _device_, preferably a /dev entry, for the parameter of the -d flag.");
		}
		if (! bkpinfo->restore_data) {
			if (!find_home_of_exe("growisofs")) {
				fatal_error("Please install growisofs (probably part of dvd+rw-tools). If you want DVD support, you need it.");
			}
			if (!find_home_of_exe("dvd+rw-format")) {
				fatal_error("Please install dvd+rw-format (probably part of dvd+rw-tools). If you want DVD support, you need it.");
			}
			if (!flag_set['s']) {
				sprintf(flag_val['s'], "%d", DEFAULT_DVD_DISK_SIZE);	// 4.7 salesman's GB = 4.482 real GB = 4582 MB
				strcat(flag_val['s'], "m");
				log_to_screen("You did not specify a size (-s) for DVD. I'm guessing %s.", flag_val['s']);
				flag_set['s'] = 1;
			}
		}
	}

	if (flag_set['t'] || flag_set['u']) {	/* tape size */
		if (strchr(flag_val['d'], ',')) {
			fatal_error("Please don't give a SCSI node. Give a _device_, preferably a /dev entry, for the parameter of the -d flag.");
		}
		if ((flag_set['O']) && (! bkpinfo->restore_data)) {
			if (flag_set['s']) {
				if (flag_set['t']) {
					fatal_error("For the moment, please don't specify a tape size. Mondo should handle end-of-tape gracefully anyway.");
				}
				if (process_the_s_switch(flag_val['s'])) {
					fatal_error("Bad -s switch");
				}
			} else if (flag_set['u'] || flag_set['t']) {
				bkpinfo->media_size = 0;
			} else {
				retval++;
				log_to_screen("Tape size not specified.\n");
			}
		}
	} else if (! bkpinfo->restore_data) {			/* CD|USB size */
		if (flag_set['s']) {
			if (process_the_s_switch(flag_val['s'])) {
				fatal_error("Bad -s switch");
			}
		}
		if (flag_set['w']) {
			bkpinfo->wipe_media_first = TRUE;
		}						/* CD-RW */
	}

	if (flag_set['n']) {
		mr_asprintf(bkpinfo->netfs_mount, "%s", flag_val['n']);
		if (!flag_set['d']) {
			mr_asprintf(bkpinfo->netfs_remote_dir, "/");
			mr_asprintf(bkpinfo->isodir, "%s", ".");
		}
		/* test for protocol */
		p = strstr(bkpinfo->netfs_mount, "://");
		if (p == NULL) {
			/* protocol not found assuming NFS for compatibility */
			mr_asprintf(q,"nfs");

			p = strchr(bkpinfo->netfs_mount, ':');
			if (p == NULL) {
				fatal_error("No protocol specified for remote share mount, nor any old NFS syntax found.\nPlease do man mondoarchive as syntax changed");
			}
			/* p points on to the string server:/path */
			p = bkpinfo->netfs_mount;

		} else {
			/* Isolate the protocol */
			*p = '\0';
			mr_asprintf(q,"%s",bkpinfo->netfs_mount);

			/* Skip proto now */
			p++;
			p++;
			p++;
		}
		/* whatever done before proto is pointed to by q */
		bkpinfo->netfs_proto = q;

		/* p points on to the string server:/path */
		/* Store the 2 values */
		q2 = bkpinfo->netfs_mount;
		mr_asprintf(bkpinfo->netfs_mount, "%s", p);
		mr_free(q2);

		/* test if we specified a user */
		p = strchr(bkpinfo->netfs_mount, '@');
		if (p != NULL) {
			/* User found. Store the 2 values */
			mr_asprintf(bkpinfo->netfs_user, "%s", bkpinfo->netfs_mount);
			p = strchr(bkpinfo->netfs_user, '@');
			*p = '\0';
			p = strchr(bkpinfo->netfs_mount, '@');
			p++;
			/* new netfs mount */
			q2 = bkpinfo->netfs_mount;
			mr_asprintf(bkpinfo->netfs_mount, "%s", p);
			mr_free(q2);
		}
		if (bkpinfo->netfs_user != NULL) {
			mr_asprintf(tmp1, "mount | grep -E \"^[a-z]*#*[%s@]*%s[/]* .*\" | cut -d' ' -f3", bkpinfo->netfs_user, bkpinfo->netfs_mount);
		} else {
			mr_asprintf(tmp1, "mount | grep -E \"^[a-z]*#*%s[/]* .*\" | cut -d' ' -f3", bkpinfo->netfs_mount);
		}
		mr_free(bkpinfo->isodir);
		bkpinfo->isodir = call_program_and_get_last_line_of_output(tmp1);
		mr_free(tmp1);

		log_msg(3, "proto = %s", bkpinfo->netfs_proto);
		log_msg(3, "mount = %s", bkpinfo->netfs_mount);
		if (bkpinfo->netfs_user) {
			log_msg(3, "user = %s", bkpinfo->netfs_user);
		}
		log_msg(3, "isodir= %s", bkpinfo->isodir);

		if (strlen(bkpinfo->isodir) < 3) {
			log_to_screen("Network share %s is not mounted. Trying to mount it for you.\n",bkpinfo->netfs_mount);
			if (bkpinfo->netfs_user) {
				if (strstr(bkpinfo->netfs_proto, "sshfs")) {
					mr_asprintf(tmp1, "sshfs %s@%s", bkpinfo->netfs_user, bkpinfo->netfs_mount);
				} else if (strstr(bkpinfo->netfs_proto, "smbfs")) {
					mr_asprintf(tmp1, "mount -t cifs %s -o user=%s", bkpinfo->netfs_mount, bkpinfo->netfs_user);
				} else if (strstr(bkpinfo->netfs_proto, "nfs")) {
					mr_asprintf(tmp1, "mount -t %s %s@%s", bkpinfo->netfs_proto, bkpinfo->netfs_user, bkpinfo->netfs_mount);
				} else {
					log_to_screen("Protocol %s not supported yet for network backups.\n", bkpinfo->netfs_proto);
					fatal_error("Bad Protocol\n");
				}
			} else {
				if (strstr(bkpinfo->netfs_proto, "sshfs")) {
					mr_asprintf(tmp1, "sshfs %s", bkpinfo->netfs_mount);
				} else if (strstr(bkpinfo->netfs_proto, "smbfs")) {
					mr_asprintf(tmp1, "mount -t cifs %s", bkpinfo->netfs_mount);
				} else if (strstr(bkpinfo->netfs_proto, "nfs")) {
					mr_asprintf(tmp1, "mount -t %s %s", bkpinfo->netfs_proto, bkpinfo->netfs_mount);
				} else {
					log_to_screen("Protocol %s not supported yet for network backups.\n", bkpinfo->netfs_proto);
					fatal_error("Bad Protocol\n");
				}
			}
			i = system(tmp1);
			mr_free(tmp1);

			if (i) {
				log_to_screen("Unable to mount Network share %s. Please mount manually.\n", bkpinfo->netfs_mount);
				retval++;
			} else {
				if (bkpinfo->netfs_user) {
					mr_asprintf(tmp1, "mount | grep -E \"^[%s@]*%s[/]* .*\" | cut -d' ' -f3", bkpinfo->netfs_user, bkpinfo->netfs_mount);
				} else {
					mr_asprintf(tmp1, "mount | grep -E \"^%s[/]* .*\" | cut -d' ' -f3", bkpinfo->netfs_mount);
				}
				mr_free(bkpinfo->isodir);
				bkpinfo->isodir = call_program_and_get_last_line_of_output(tmp1);
				if (strlen(bkpinfo->isodir) < 3) {
					retval++;
					log_to_screen("Network share %s is strangely not mounted. Please mount manually...\n", bkpinfo->netfs_mount);
				}
			}
		}
	}

	if (flag_set['c']) {
		bkpinfo->backup_media_type = cdr;
	}
	if (flag_set['C']) {
		bkpinfo->backup_media_type = cdstream;
	}
	if (flag_set['i']) {
		bkpinfo->backup_media_type = iso;
	}
	if (flag_set['n']) {
		bkpinfo->backup_media_type = netfs;
		/* Never try to eject a Network device */
		bkpinfo->please_dont_eject = TRUE;
	}
	if (flag_set['r']) {
		bkpinfo->backup_media_type = dvd;
	}
	if (flag_set['t']) {
		bkpinfo->backup_media_type = tape;
	}
	if (flag_set['u']) {
		bkpinfo->backup_media_type = udev;
	}
	if (flag_set['w']) {
		bkpinfo->backup_media_type = cdrw;
	}
	if (flag_set['U']) {
		bkpinfo->backup_media_type = usb;
		/* Never try to eject a USB device */
		bkpinfo->please_dont_eject = TRUE;
	}
	if (flag_set['z']) {
		if (find_home_of_exe("getfattr")) {
			mr_asprintf(g_getfattr,"getfattr");
		}
		if (find_home_of_exe("getfacl")) {
			mr_asprintf(g_getfacl,"getfacl");
		}
	}

	/* optional, popular */
	if (flag_set['g']) {
		g_text_mode = FALSE;
	}

	if (flag_set['E']) {
		if ((flag_val['E'][0] == '-')) {
			retval++;
			log_to_screen("Please supply a sensible value with '-E'\n");
		}
		mr_asprintf(tmp1, "%s", flag_val['E']);

		p = tmp1;
		q = tmp1;

		/* Cut the flag_val['E'] in parts containing all paths to test them */
		while (p != NULL) {
			q = strchr(p, '|');
			if (q != NULL) {
				*q = '\0';
				/* Fix bug 14 where ending / cause a problem later 
				* so handled here for the moment */
				q--;
				if (*q == '/') {
					*q = '\0';
				}
				q++;
				/* End of bug fix */
				if ((stat(p, &buf) != 0) && (! bkpinfo->restore_data)) {
					log_msg(1, "WARNING ! %s doesn't exist", p);
				}
				p = q+1 ;
			} else {
				if ((stat(p, &buf) != 0) && (! bkpinfo->restore_data)) {
					log_msg(1, "WARNING ! %s doesn't exist", p);
				}
				p = NULL;
			}
		}
		mr_free(tmp1);

		mr_make_devlist_from_pathlist(flag_val['E'], 'E');
		log_msg(4, "Finished with the -E option");
	}

	if (flag_set['e']) {
		bkpinfo->please_dont_eject = TRUE;
	}

	if (flag_set['M']) {
		g_max_biggie_size = atol(flag_val['M']);
		log_msg(1, "Max size for biggie file is now %ld KB", g_max_biggie_size);
	}

	if ((flag_set['N'])	&& (! bkpinfo->restore_data))		// exclude Network mounts & devices
	{
		psz = list_of_NETFS_mounts_only();
		log_msg(5, "-N means we'll exclude %s", psz);
		if (bkpinfo->exclude_paths) {
			mr_strcat(bkpinfo->exclude_paths, "|%s", psz);
			mr_free(psz);
		} else {
			bkpinfo->exclude_paths = psz;
		}

		if (bkpinfo->exclude_paths != NULL) {
			log_msg(3, "-N means we're now excluding %s", bkpinfo->exclude_paths);
		}
	}

	if (flag_set['b']) {
		mr_asprintf(psz, "%s", flag_val['b']);
		log_msg(1, "psz = '%s'", psz);
		if (psz[strlen(psz) - 1] == 'k') {
			psz[strlen(psz) - 1] = '\0';
			itbs = atol(psz) * 1024L;
		} else {
			itbs = atol(psz);
		}
		mr_free(psz);
		log_msg(1, "'%s' --> %ld", flag_val['b'], itbs);
		log_msg(1, "Internal tape block size is now %ld bytes", itbs);
		if (itbs % 512 != 0 || itbs < 256 || itbs > 1024L * 1024) {
			fatal_error("Are you nuts? Silly, your internal tape block size is. Abort, I shall.");
		}
		bkpinfo->internal_tape_block_size = itbs;
	}

	if ((flag_set['D']) && (! bkpinfo->restore_data)) {
		bkpinfo->differential = 1;
//      bkpinfo->differential = atoi (flag_val['D']);
		if ((bkpinfo->differential < 1) || (bkpinfo->differential > 9)) {
			fatal_error("The D option should be between 1 and 9 inclusive");
		}
	}

	if (flag_set['x']) {
		mr_asprintf(bkpinfo->image_devs, "%s", flag_val['x']);
		if ((run_program_and_log_output("which ntfsclone", 2)) && (! bkpinfo->restore_data)) {
			fatal_error("Please install ntfsprogs package/tarball.");
		}
	}

	if (flag_set['m']) {
		bkpinfo->manual_cd_tray = TRUE;
	}

	if ((flag_set['k']) && (! bkpinfo->restore_data)) {
		mr_asprintf(bkpinfo->kernel_path, "%s", flag_val['k']);
		if (!does_file_exist(bkpinfo->kernel_path)) {
			retval++;
			log_to_screen("You specified kernel '%s', which does not exist\n", bkpinfo->kernel_path);
		}
	}

	if (flag_set['p']) {
		mr_asprintf(bkpinfo->prefix, "%s", flag_val['p']);
		log_msg(1,"Prefix forced to %s",bkpinfo->prefix);
	}

	if (flag_set['d']) {		/* backup directory (if ISO/NETFS) */
		if (flag_set['i']) {
			mr_asprintf(bkpinfo->isodir, "%s", flag_val['d']);
			sprintf(tmp, "ls -l %s", bkpinfo->isodir);
			if (run_program_and_log_output(tmp, 2)) {
				fatal_error("output folder does not exist - please create it");
			}
		} else if (flag_set['n']) {
			mr_free(bkpinfo->netfs_remote_dir);
			mr_asprintf(bkpinfo->netfs_remote_dir, "%s", flag_val['d']);
		} else {				/* backup device (if tape/CD-R/CD-RW) */
			mr_asprintf(bkpinfo->media_device, "%s", flag_val['d']);
		}
	}

	if ((flag_set['n']) && (! bkpinfo->restore_data)) {
		mr_asprintf(tmp1,"%s/%s/.dummy.txt", bkpinfo->isodir,bkpinfo->netfs_remote_dir);
		if ((bkpinfo->netfs_user) && (strstr(bkpinfo->netfs_proto,"nfs"))) {
			mr_asprintf(tmp2, "su - %s -c \"echo hi > %s\"", bkpinfo->netfs_user, tmp1);
		} else {
			mr_asprintf(tmp2, "echo hi > %s", tmp1);
		}
		i = run_program_and_log_output(tmp2, 2);
		mr_free(tmp2);

		if (i) {
			retval++;
			log_to_screen("Are you sure directory '%s' exists in remote dir '%s'?\nIf so, do you have rights to write to it?\n", bkpinfo->netfs_remote_dir, bkpinfo->netfs_mount);
		}
		unlink(tmp1);
		mr_free(tmp1);
	}

	if (!flag_set['d'] && (flag_set['c'] || flag_set['w'] || flag_set['C'])) {
		if (g_kernel_version >= 2.6) {
			tmp1 = mr_popup_and_get_string("Device", "Please specify the device", bkpinfo->media_device);
			if (tmp1 == NULL) {
				retval++;
				log_to_screen("User opted to cancel.");
			} else {
				mr_free(bkpinfo->media_device);
				bkpinfo->media_device = tmp1;
			}
		} else if (find_cdrw_device(bkpinfo->media_device)) {
			retval++;
			log_to_screen("Tried and failed to find CD-R[W] drive automatically.\n");
		} else {
			flag_set['d'] = TRUE;
			strncpy(flag_val['d'], bkpinfo->media_device, MAX_STR_LEN / 4);
		}
	}

	if ((!flag_set['d'] && !flag_set['n'] && !flag_set['C']) && (! bkpinfo->restore_data)) {
		retval++;
		log_to_screen("Please specify the backup device/directory.\n");
		fatal_error("You didn't use -d to specify the backup device/directory.");
	}

	for (i = '0'; i <= '9'; i++) {
		if (flag_set[i]) {
			bkpinfo->compression_level = i - '0';
		}						/* not '\0' but '0' */
	}

	if (flag_set['S']) {
		setup_scratchdir(flag_val['S']);
		mr_asprintf(tmp1, "touch %s/.foo.dat", bkpinfo->scratchdir);
		if (run_program_and_log_output(tmp1, 1)) {
			retval++;
			mr_free(tmp1);
			log_to_screen("Please specify a scratchdir which I can write to. :)");
			fatal_error("I cannot write to the scratchdir you specified.");
		}
		mr_free(tmp1);

		mr_asprintf(tmp1, "ln -sf %s/.foo.dat %s/.bar.dat", bkpinfo->scratchdir, bkpinfo->scratchdir);
		if (run_program_and_log_output(tmp1, 1)) {
			retval++;
			mr_free(tmp1);
			log_to_screen("Please don't specify a SAMBA or VFAT or NFS scratchdir.");
			fatal_error("I cannot write to the scratchdir you specified.");
		}
		mr_free(tmp1);
	}

	if (flag_set['T']) {
		setup_tmpdir(flag_val['T']);
		mr_asprintf(tmp1, "touch %s/.foo.dat", bkpinfo->tmpdir);
		i = run_program_and_log_output(tmp1, 1);
		mr_free(tmp1);

		if (i) {
			retval++;
			log_to_screen("Please specify a tempdir which I can write to. :)");
			fatal_error("I cannot write to the tempdir you specified.");
		}
		mr_asprintf(tmp1, "ln -sf %s/.foo.dat %s/.bar.dat", bkpinfo->tmpdir, bkpinfo->tmpdir);
		i = run_program_and_log_output(tmp1, 1);
		mr_free(tmp1);

		if (i) {
			retval++;
			log_to_screen("Please don't specify a SAMBA or VFAT or NFS tmpdir.");
			fatal_error("I cannot write to the tempdir you specified.");
		}
	}

	if ((flag_set['A']) && (! bkpinfo->restore_data)) {
		mr_asprintf(bkpinfo->call_after_iso, "%s", flag_val['A']);
	}

	if ((flag_set['B']) && (! bkpinfo->restore_data)) {
		mr_asprintf(bkpinfo->call_before_iso, "%s", flag_val['B']);
	}

	if ((flag_set['H']) && (! bkpinfo->restore_data)) {
		g_cd_recovery = TRUE;
	}

	if ((flag_set['l']) && (! bkpinfo->restore_data)) {
#ifdef __FreeBSD__
#  define BOOT_LOADER_CHARS "GLBMR"
#else
#  ifdef __IA64__
#    define BOOT_LOADER_CHARS "GER"
#  else
#    define BOOT_LOADER_CHARS "GLR"
#  endif
#endif
		if (!strchr(BOOT_LOADER_CHARS, (bkpinfo->boot_loader = flag_val['l'][0]))) {
			log_msg(1, "%c? What is %c? I need G, L, E or R.", bkpinfo->boot_loader, bkpinfo->boot_loader);
			fatal_error("Please specify GRUB, LILO, ELILO  or RAW with the -l switch");
		}
#undef BOOT_LOADER_CHARS
	}

	if (flag_set['f']) {
		mr_free(bkpinfo->boot_device);
		mr_asprintf(bkpinfo->boot_device, "%s", resolve_softlinks_to_get_to_actual_device_file(flag_val['f']));
	}

	if ((flag_set['P']) && (! bkpinfo->restore_data)) {
		mr_asprintf(bkpinfo->postnuke_tarball, "%s", flag_val['P']);
	}

	if (flag_set['Q']) {
		i = which_boot_loader(tmp);
		log_msg(3, "boot loader is %c, residing at %s", i, tmp);
		printf("boot loader is %c, residing at %s\n", i, tmp);
		finish(0);
	}

	if ((flag_set['L']) && (! bkpinfo->restore_data)) {
		bkpinfo->use_lzo = TRUE;
		if (run_program_and_log_output("which lzop", 2)) {
			retval++;
			log_to_screen("Please install LZOP. You can't use '-L' until you do.\n");
		}
	}

	if (flag_set['F']) {
		log_msg(3, "-F means we will fail immediately at the first interaction request");
		g_fail_immediately = TRUE;
	}

	if ((flag_set['G']) && (! bkpinfo->restore_data)) {
		bkpinfo->use_gzip = TRUE;
		if (run_program_and_log_output("which gzip", 2)) {
			retval++;
			log_to_screen("Please install gzip. You can't use '-G' until you do.\n");
		}
	}

	if ((flag_set['Y']) && (! bkpinfo->restore_data)) {
		bkpinfo->use_lzma = TRUE;
		if (run_program_and_log_output("which lzma", 2)) {
			retval++;
			log_to_screen("Please install lzma. You can't use '-Y' until you do.\n");
		}
	}

	bkpinfo->use_obdr = FALSE;
	if (flag_set['o']) {
		if ((!flag_set['t']) && (! bkpinfo->restore_data)) {
			log_to_screen("OBDR support is only available for tapes. Use the -t option");
			fatal_error("Aborting");
		}
		bkpinfo->use_obdr = TRUE;
	}

#ifndef __FreeBSD__
	if ((!is_this_a_valid_disk_format("vfat")) && (! bkpinfo->restore_data)) {
		bkpinfo->make_cd_use_lilo = TRUE;
		log_to_screen("Your kernel appears not to support vfat filesystems. I am therefore");
		log_to_screen("using LILO instead of SYSLINUX as the media boot loader.");
	}
	if ((run_program_and_log_output("which mkfs.vfat", 2)) && (! bkpinfo->restore_data)) {
		bkpinfo->make_cd_use_lilo = TRUE;
#ifdef __IA32__
		log_to_screen("Your filesystem is missing 'mkfs.vfat', so I cannot use SYSLINUX as");
		log_to_screen("your boot loader. I shall therefore use LILO instead.");
#endif
#ifdef __IA64__
		log_to_screen("Your filesystem is missing 'mkfs.vfat', so I cannot prepare the EFI");
		log_to_screen("environment correctly. Please install it.");
		fatal_error("Aborting");
#endif
	}
#ifdef __IA64__
	/* We force ELILO usage on IA64 */
	bkpinfo->make_cd_use_lilo = TRUE;
#endif
#endif

	if (! bkpinfo->restore_data) {
		i = flag_set['O'] + flag_set['V'];
		if (i == 0) {
			retval++;
			log_to_screen("Specify backup (-O), verify (-V) or both (-OV).\n");
		}
	}

	if ((! bkpinfo->restore_data) && (flag_set['Z'])) {
			fatal_error("The -Z switch is only valid in restore mode");
	}

	if (flag_set['Z']) {
		if (! strcmp(flag_val['Z'], "nuke")) {
			bkpinfo->restore_mode = nuke;
		} else if (! strcmp(flag_val['Z'], "interactive")) {
			bkpinfo->restore_mode = interactive;
		} else if (! strcmp(flag_val['Z'], "compare")) {
			bkpinfo->restore_mode = compare;
		} else if (! strcmp(flag_val['Z'], "mbr")) {
			bkpinfo->restore_mode = mbr;
		} else if (! strcmp(flag_val['Z'], "iso")) {
			bkpinfo->restore_mode = isoonly;
		} else if (! strcmp(flag_val['Z'], "isonuke")) {
			bkpinfo->restore_mode = isonuke;
		} else {
			bkpinfo->restore_mode = interactive;
		}
	}

/* and finally... */

	paranoid_free(tmp);
	return (retval);
}



/**
 * Get the switches from @p argc and @p argv using getopt() and place them in
 * @p flag_set and @p flag_val.
 * @param argc The argument count (@p argc passed to main()).
 * @param argv The argument vector (@p argv passed to main()).
 * @param flag_val An array indexed by switch letter - if a switch is set and
 * has an argument then set flag_val[switch] to that argument.
 * @param flag_set An array indexed by switch letter - if a switch is set then
 * set flag_set[switch] to TRUE, else set it to FALSE.
 * @return The number of problems with the command line (0 for success).
 */
int retrieve_switches_from_command_line(int argc, char *argv[], char flag_val[128][MAX_STR_LEN], bool flag_set[128])
{
	/*@ ints ** */
	int opt = 0;
	int i = 0;
	int len;

	/*@ bools *** */
	bool bad_switches = FALSE;

	assert(flag_val != NULL);
	assert(flag_set != NULL);

	for (i = 0; i < 128; i++) {
		flag_val[i][0] = '\0';
		flag_set[i] = FALSE;
	}
	while ((opt = getopt(argc, argv, MONDO_OPTIONS)) != -1) {
		if (opt == '?') {
			bad_switches = TRUE;
		} else {
			if (flag_set[opt]) {
				bad_switches = TRUE;
				log_to_screen("Switch -%c previously defined as %s\n", opt, flag_val[opt]);
			} else {
				flag_set[opt] = TRUE;
				if (optarg) {
					len = strlen(optarg);
					if (optarg[0] != '/' && optarg[len - 1] == '/') {
						optarg[--len] = '\0';
						log_to_screen("Warning - param '%s' should not have trailing slash!", optarg);
					}
					if (opt == 'd') {
						if (strchr(flag_val[opt], '/')
							&& flag_val[opt][0] != '/') {
							log_to_screen("-%c flag --- must be absolute path --- '%s' isn't absolute", opt, flag_val[opt]);
							bad_switches = TRUE;
						}
					}
					strcpy(flag_val[opt], optarg);
				}
			}
		}
	}
	for (i = optind; i < argc; i++) {
		bad_switches = TRUE;
		log_to_screen("Invalid arg -- %s\n", argv[i]);
	}
	return (bad_switches);
}




/**
 * Print a not-so-helpful help message and exit.
 */
void help_screen()
{
	log_msg(1, "Type 'man mondoarchive' for more information");
	exit(1);
}


/**
 * Terminate Mondo in response to a signal.
 * @param sig The signal number received.
 */
void terminate_daemon(int sig)
{
	char *tmp = NULL;
	char *tmp2 = NULL;

	switch (sig) {
	case SIGINT:
		mr_asprintf(tmp, "SIGINT");
		mr_asprintf(tmp2, "You interrupted me :-)");
		break;
	case SIGKILL:
		mr_asprintf(tmp, "SIGKILL");
		mr_asprintf(tmp2, "I seriously have no clue how this signal even got to me. Something's wrong with your system.");
		break;
	case SIGTERM:
		mr_asprintf(tmp, "SIGTERM");
		mr_asprintf(tmp2, "Got terminate signal");
		break;
	case SIGHUP:
		mr_asprintf(tmp, "SIGHUP");
		mr_asprintf(tmp2, "Hangup on line");
		break;
	case SIGSEGV:
		mr_asprintf(tmp, "SIGSEGV");
		mr_asprintf(tmp2, "Internal programming error. Please send a backtrace as well as your log.");
		break;
	case SIGPIPE:
		mr_asprintf(tmp, "SIGPIPE");
		mr_asprintf(tmp2, "Pipe was broken");
		break;
	case SIGABRT:
		mr_asprintf(tmp, "SIGABRT");
		mr_asprintf(tmp2, "Abort - probably failed assertion. I'm sleeping for a few seconds so you can read the message.");
		break;
	default:
		mr_asprintf(tmp, "(Unknown)");
		mr_asprintf(tmp2, "(Unknown)");
	}

	mr_strcat(tmp, " signal received from OS");
	log_to_screen(tmp);
	mr_free(tmp);

	log_to_screen(tmp2);
	mr_free(tmp2);
	if (sig == SIGABRT) {
		sleep(10);
	}
	kill_buffer();

	free_MR_global_filenames();

	fatal_error("MondoRescue is terminating in response to a signal from the OS");
	finish(254);				// just in case
}




/**
 * Turn signal-trapping on or off.
 * @param on If TRUE, turn it on; if FALSE, turn it off (we still trap it, just don't do as much).
 */
void set_signals(int on)
{
	int signals[] = { SIGTERM, SIGHUP, SIGTRAP, SIGABRT, SIGINT, SIGKILL, SIGSTOP, 0 };
	int i;

	signal(SIGPIPE, sigpipe_occurred);
	for (i = 0; signals[i]; i++) {
		if (on) {
			signal(signals[i], terminate_daemon);
		} else {
			signal(signals[i], termination_in_progress);
		}
	}
}




/**
 * Exit immediately without cleaning up.
 * @param sig The signal we are exiting due to.
 */
void termination_in_progress(int sig)
{
	log_msg(1, "Termination in progress");
	usleep(1000);
	pthread_exit(0);
}

/* @} - end of cliGroup */
