/* libmondo-devices.c                 Subroutines for handling devices
   $Id: libmondo-devices.c 2326 2009-08-18 13:19:54Z bruno $
*/

/**
 * @file
 * Functions to handle interactions with backup devices.
 */

#include "my-stuff.h"
#include "mr_mem.h"
#include "mondostructures.h"
#include "libmondo-files-EXT.h"
#include "libmondo-devices.h"
#include "lib-common-externs.h"
#include "libmondo-string-EXT.h"
#include "libmondo-tools-EXT.h"
#include "libmondo-gui-EXT.h"
#include "libmondo-fork-EXT.h"
#include "libmondo-stream-EXT.h"

extern void mr_strip_spaces(char *);

#include <sys/types.h>
#ifdef __FreeBSD__
#define DKTYPENAMES
#define FSTYPENAMES
#include <sys/disklabel.h>
#include <sys/disk.h>
#elif linux
#define u64 unsigned long long
#include <linux/fs.h>			/* for BLKGETSIZE64 */
#include <linux/hdreg.h>
#endif

/*@unused@*/
//static char cvsid[] = "$Id: libmondo-devices.c 2326 2009-08-18 13:19:54Z bruno $";

extern int g_current_media_number;
extern double g_kernel_version;

extern bool g_ISO_restore_mode;
extern char *g_selfmounted_isodir;
extern char *MONDO_LOGFILE;

static char g_cdrw_drive_is_here[MAX_STR_LEN / 4] = "";
static char g_cdrom_drive_is_here[MAX_STR_LEN / 4] = "";
static char g_dvd_drive_is_here[MAX_STR_LEN / 4] = "";


/**
 * ????? @bug ?????
 * @ingroup globalGroup
 */
bool g_restoring_live_from_cd = FALSE;
bool g_restoring_live_from_nfs = FALSE;

extern t_bkptype g_backup_media_type;	// set by main()

/* Reference to global bkpinfo */
extern struct s_bkpinfo *bkpinfo;




void set_g_cdrom_and_g_dvd_to_bkpinfo_value() {

	if (bkpinfo->media_device) {
		strcpy(g_cdrom_drive_is_here, bkpinfo->media_device);	// just in case
		strcpy(g_dvd_drive_is_here, bkpinfo->media_device);	// just in case
	}
}



/**
 * Retract all CD trays and wait for autorun to complete.
 * @ingroup deviceGroup
 */
void retract_CD_tray_and_defeat_autorun(void)
{
//  log_it("rctada: Retracting all CD trays", __LINE__);
	if (strlen(g_cdrom_drive_is_here) > 0) {
		inject_device(g_cdrom_drive_is_here);
	}
	if (strlen(g_dvd_drive_is_here) > 0) {
		inject_device(g_dvd_drive_is_here);
	}
	if (strlen(g_cdrw_drive_is_here) > 0) {
		inject_device(g_cdrw_drive_is_here);
	}
//  log_it("rctada: killing autorun");
//  run_program_and_log_output("killall autorun", TRUE);
	if (!run_program_and_log_output("ps | grep autorun | grep -v grep", 5)) {
		log_it("autorun detected; sleeping for 2 seconds");
		sleep(2);
	}
	log_it("rctada: Unmounting all CD drives", __LINE__);
	run_program_and_log_output("umount /dev/cdr* /dev/dvd*", 5);
}



/**
 * Determine whether we're booted off a ramdisk.
 * @return @c TRUE (we are) or @c FALSE (we aren't).
 * @ingroup utilityGroup
 */
bool am_I_in_disaster_recovery_mode(void)
{
	char *tmp = NULL;
	bool is_this_a_ramdisk = FALSE;

	mr_asprintf(tmp, "%s", where_is_root_mounted());
	log_msg(0, "root is currently mounted at %s\n", tmp);

#ifdef __FreeBSD__
	if (strstr(tmp, "/dev/md")) {
		is_this_a_ramdisk = TRUE;
	}
#else
	if (!strncmp(tmp, "/dev/ram", 8)
		|| (!strncmp(tmp, "/dev/rd", 7) && !strcmp(tmp, "/dev/rd/")
			&& strncmp(tmp, "/dev/rd/cd", 10)) || strstr(tmp, "rootfs")
		|| !strcmp(tmp, "/dev/root")) {
		is_this_a_ramdisk = TRUE;
	} else {
		is_this_a_ramdisk = FALSE;
	}
#endif
	mr_free(tmp);

	if (is_this_a_ramdisk) {
		if (!does_file_exist("/THIS-IS-A-RAMDISK")) {
			log_to_screen("Using /dev/root is stupid of you but I'll forgive you.");
			is_this_a_ramdisk = FALSE;
		}
	}
	if (does_file_exist("/THIS-IS-A-RAMDISK")) {
		is_this_a_ramdisk = TRUE;
	}
	log_msg(1, "Is this a ramdisk? result = %s", (is_this_a_ramdisk) ? "TRUE" : "FALSE");
	return (is_this_a_ramdisk);
}





/**
 * Turn @c bkpinfo->backup_media_type into a human-readable string.
 * @return The human readable string (e.g. @c cdr becomes <tt>"cdr"</tt>).
 * @note The returned string points to static storage that will be overwritten with each call.
 * @ingroup stringGroup
 */
static char *bkptype_to_string(t_bkptype bt)
{
	static char output[MAX_STR_LEN / 4];
	switch (bt) {
	case none:
		strcpy(output, "none");
		break;
	case iso:
		strcpy(output, "iso");
		break;
	case cdr:
		strcpy(output, "cdr");
		break;
	case cdrw:
		strcpy(output, "cdrw");
		break;
	case cdstream:
		strcpy(output, "cdstream");
		break;
	case nfs:
		strcpy(output, "nfs");
		break;
	case tape:
		strcpy(output, "tape");
		break;
	case udev:
		strcpy(output, "udev");
		break;
	case usb:
		strcpy(output, "usb");
		break;
	default:
		strcpy(output, "default");
	}
	return (output);
}



/**
 * @addtogroup deviceGroup
 * @{
 */
/**
 * Eject the tray of the specified CD device.
 * @param dev The device to eject.
 * @return the return value of the @c eject command. (0=success, nonzero=failure)
 */
int eject_device(char *dev)
{
	char *command = NULL;
	int res1 = 0, res2 = 0;

	if (dev == NULL) {
		return (1);
	}

	if (IS_THIS_A_STREAMING_BACKUP(g_backup_media_type)
		&& g_backup_media_type != udev) {
		mr_asprintf(command, "mt -f %s offline", dev);
		res1 = run_program_and_log_output(command, 1);
		mr_free(command);
	} else {
		res1 = 0;
	}

#ifdef __FreeBSD__
	if (strstr(dev, "acd")) {
		mr_asprintf(command, "cdcontrol -f %s eject", dev);
	} else {
		mr_asprintf(command, "camcontrol eject `echo %s | sed 's|/dev/||'`", dev);
	}
#else
	mr_asprintf(command, "eject %s", dev);
#endif

	log_msg(3, "Ejecting %s", dev);
	res2 = run_program_and_log_output(command, 1);
	mr_free(command);
	if (res1 && res2) {
		return (1);
	} else {
		return (0);
	}
}

/**
 * Load (inject) the tray of the specified CD device.
 * @param dev The device to load/inject.
 * @return 0 for success, nonzero for failure.
 */
int inject_device(char *dev)
{
	char *command = NULL;
	int i;

	if (dev == NULL) {
		return (1);
	}

#ifdef __FreeBSD__
	if (strstr(dev, "acd")) {
		mr_asprintf(command, "cdcontrol -f %s close", dev);
	} else {
		mr_asprintf(command, "camcontrol load `echo %s | sed 's|/dev/||'`", dev);
	}
#else
	mr_asprintf(command, "eject -t %s", dev);
#endif
	i = run_program_and_log_output(command, FALSE);
	mr_free(command);
	return (i);
}


/**
 * Determine whether the specified @p device (really, you can use any file)
 * exists.
 * @return TRUE if it exists, FALSE if it doesn't.
 */
bool does_device_exist(char *device)
{

	/*@ buffers *********************************************************** */
	char *tmp = NULL;
	bool ret;

	assert_string_is_neither_NULL_nor_zerolength(device);

	mr_asprintf(tmp, "ls %s > /dev/null 2> /dev/null", device);

	if (system(tmp)) {
		ret = FALSE;
	} else {
		ret = TRUE;
	}
	mr_free(tmp);
	return (ret);
}


/**
 * Determine whether a non-Microsoft partition exists on any connected hard drive.
 * @return TRUE (there's a Linux/FreeBSD partition) or FALSE (Microsoft has taken over yet another innocent PC).
 */
bool does_nonMS_partition_exist(void)
{
#if __FreeBSD__
	return
		!system
		("for drive in /dev/ad? /dev/da?; do fdisk $drive | grep -q FreeBSD && exit 0; done; false");
#else
	return
		!system
		("parted2fdisk -l 2>/dev/null | grep '^/dev/' | grep -Eqv '(MS|DOS|FAT|NTFS)'");
#endif
}

/**
 * Determine whether the specified @p partno exists on the specified @p drive.
 * @param drive The drive to search for the partition in.
 * @param partno The partition number to look for.
 * @return 0 if it exists, nonzero otherwise.
 */
int does_partition_exist(const char *drive, int partno)
{
	/*@ buffers **************************************************** */
	char *program = NULL;
	char *incoming;
	char *searchstr = NULL;
	char *tmp = NULL;

	/*@ ints ******************************************************* */
	int res = 0;

	/*@ pointers *************************************************** */
	FILE *fin;


	/*@ end vars *************************************************** */
	assert_string_is_neither_NULL_nor_zerolength(drive);
	assert(partno >= 0 && partno < 999);

	malloc_string(incoming);

#ifdef __FreeBSD__
	// We assume here that this is running from mondorestore. (It is.)
	tmp = build_partition_name(drive, partno);
	mr_asprintf(program, "ls %s %s >/dev/null 2>&1", drive, tmp);
	mr_free(tmp);

	res = system(program);
	mr_free(program);
	return (res);
#else
	/* To avoid compiler warnings */
	tmp = NULL;
#endif

	mr_asprintf(program, "parted2fdisk -l %s 2> /dev/null", drive);
	fin = popen(program, "r");
	if (!fin) {
		log_it("program=%s", program);
		log_OS_error("Cannot popen-in program");
		mr_free(program);
		return (0);
	}
	mr_free(program);

	searchstr = build_partition_name(drive, partno);
	mr_strcat(searchstr, " ");
	for (res = 0; !res && fgets(incoming, MAX_STR_LEN - 1, fin);) {
		if (strstr(incoming, searchstr)) {
			res = 1;
		}
	}
	mr_free(searchstr);

	if (pclose(fin)) {
		log_OS_error("Cannot pclose fin");
	}
	paranoid_free(incoming);
	return (res);
}





/**
 * Determine whether given NULL-terminated @p str exists in the MBR of @p dev.
 * @param dev The device to look in.
 * @param str The string to look for.
 * @return TRUE if it exists, FALSE if it doesn't.
 */
bool does_string_exist_in_boot_block(char *dev, char *str)
{
	/*@ buffers **************************************************** */
	char *command = NULL;

	/*@ end vars *************************************************** */
	int i;

	assert_string_is_neither_NULL_nor_zerolength(dev);
	assert_string_is_neither_NULL_nor_zerolength(str);

	mr_asprintf(command, "dd if=%s bs=446 count=1 2> /dev/null | strings | grep \"%s\" > /dev/null 2> /dev/null", dev, str);
	i = system(command);
	mr_free(command);
	if (i) {
		return (FALSE);
	} else {
		return (TRUE);
	}
}

/**
 * Determine whether specified @p str exists in the first @p n sectors of
 * @p dev.
 * @param dev The device to look in.
 * @param str The string to look for.
 * @param n The number of 512-byte sectors to search.
 */
bool does_string_exist_in_first_N_blocks(char *dev, char *str, int n)
{
	/*@ buffers **************************************************** */
	char *command = NULL;
	/*@ end vars *************************************************** */
	int i;

	mr_asprintf(command, "dd if=%s bs=512 count=%i 2> /dev/null | strings | grep \"%s\" > /dev/null 2> /dev/null", dev, n, str);
	i = system(command);
	mr_free(command);
	if (i) {
		return (FALSE);
	} else {
		return (TRUE);
	}
}



/**
 * Try to mount CD-ROM at @p mountpoint. If the CD-ROM is not found or has
 * not been specified, call find_cdrom_device() to find it.
 * @param mountpoint Where to mount the CD-ROM.
 * @return 0 for success, nonzero for failure.
 * @see mount_CDROM_here
 */
int find_and_mount_actual_cd(char *mountpoint) {

	/*@ buffers ***************************************************** */

	/*@ int's  ****************************************************** */
	int res;
	char *dev = NULL;
	char *p = NULL;

	/*@ end vars **************************************************** */

	assert(bkpinfo != NULL);
	assert_string_is_neither_NULL_nor_zerolength(mountpoint);

	if (g_backup_media_type == dvd) {
		if (g_dvd_drive_is_here[0]) {
			mr_asprintf(dev, "%s", g_dvd_drive_is_here);
		}
		if (dev == NULL) {
			dev = find_dvd_device();
		}
	} else {
		if (g_cdrom_drive_is_here[0]) {
			mr_asprintf(dev, "%s", g_cdrom_drive_is_here);
		}
		if (dev == NULL) {
			dev = find_cdrom_device(FALSE);
		}
	}

	if (bkpinfo->backup_media_type != iso) {
		retract_CD_tray_and_defeat_autorun();
	}

	if ((dev == NULL) || (res = mount_CDROM_here(dev, mountpoint))) {
		p = popup_and_get_string ("CD-ROM device", "Please enter your CD-ROM's /dev device",dev);
		if (p == NULL) {
			res = 1;
		} else {
			res = mount_CDROM_here(p, mountpoint);
		}
		mr_free(p);
	}
	if (res) {
		log_msg(1, "mount failed");
	} else {
		log_msg(1, "mount succeeded with %s", dev);
	}
	mr_free(dev);
	return (res);
}






/**
 * Locate a CD-R/W writer's SCSI node.
 * @param cdrw_device SCSI node will be placed here.
 * @return 0 for success, nonzero for failure.
 */
char *find_cdrw_device()
{
	/*@ buffers ************************ */
	char *tmp = NULL;
	char *cdr_exe = NULL;
	char *command = NULL;
	char *cdrw_device = NULL;

	if (g_cdrw_drive_is_here[0]) {
		mr_asprintf(cdrw_device, g_cdrw_drive_is_here);
		log_msg(3, "Been there, done that. Returning %s", cdrw_device);
		return(cdrw_device);
	}
	if (g_backup_media_type == dvd) {
		log_msg(1, "This is dumb. You're calling find_cdrw_device() but you're backing up to DVD. WTF?");
		return(NULL);
	}
	run_program_and_log_output("insmod ide-scsi", -1);
	if (find_home_of_exe("cdrecord")) {
		mr_asprintf(cdr_exe, "cdrecord");
	} else {
		mr_asprintf(cdr_exe, "dvdrecord");
	}
	if (find_home_of_exe(cdr_exe)) {
		mr_asprintf(command, "%s -scanbus 2> /dev/null | tr -s '\t' ' ' | grep \"[0-9]*,[0-9]*,[0-9]*\" | grep -v \"[0-9]*) \\*\" | grep CD | cut -d' ' -f2 | head -n1", cdr_exe);
		mr_asprintf(tmp, "%s", call_program_and_get_last_line_of_output(command));
		mr_free(command);
	}
	if ((tmp == NULL) || (strlen(tmp) < 2)) {
		mr_free(tmp);
		mr_free(cdr_exe);
		return(NULL);
	} else {
		cdrw_device = tmp;
		log_it("Found CDRW device - %s", cdrw_device);
		strcpy(g_cdrw_drive_is_here, cdrw_device);
		mr_free(cdr_exe);
		return (cdrw_device);
	}
}


/**
 * Attempt to locate a CD-ROM device's /dev entry.
 * Several different methods may be used to find the device, including
 * calling @c cdrecord, searching @c dmesg, and trial-and-error.
 * @param output Where to put the located /dev entry.
 * @param try_to_mount Whether to mount the CD as part of the test; if mount
 * fails then return failure.
 * @return 0 for success, nonzero for failure.
 */
char *find_cdrom_device(bool try_to_mount)
{
	/*@ pointers **************************************************** */
	FILE *fin;
	char *p;
	char *q;
	char *r;

	/*@ bool's ****************************************************** */
	bool found_it = FALSE;

	/*@ buffers ***************************************************** */
	char *tmp = NULL;
	char *tmp1 = NULL;
	char *output = NULL;
	char *cdr_exe = NULL;
	char *phrase_one = NULL;
	char *phrase_two = NULL;
	char *command = NULL;
#ifndef __FreeBSD__
	char *dvd_last_resort = NULL;
#endif
	char *mountpoint = NULL;
	static char the_last_place_i_found_it[MAX_STR_LEN] = "";

	/*@ intialize *************************************************** */

	output = NULL;
#ifndef __FreeBSD__
	mr_asprintf(dvd_last_resort, "%s", "");;
#endif

	/*@ end vars **************************************************** */

	if (g_cdrom_drive_is_here[0] && !isdigit(g_cdrom_drive_is_here[0])) {
		mr_asprintf(output, "%s", g_cdrom_drive_is_here);
		log_msg(3, "Been there, done that. Returning %s", output);
		return(output);
	}
	if (the_last_place_i_found_it[0] != '\0' && !try_to_mount) {
		mr_asprintf(output, "%s", the_last_place_i_found_it);
		log_msg(3, "find_cdrom_device() --- returning last found location - '%s'", output);
		return(output);
	}

	mr_asprintf(mountpoint, "%s/cd.mnt", bkpinfo->tmpdir);
	make_hole_for_dir(mountpoint);

	if (find_home_of_exe("cdrecord")) {
		mr_asprintf(cdr_exe, "cdrecord");
	} else {
		mr_asprintf(cdr_exe, "dvdrecord");
	}

	if (!find_home_of_exe(cdr_exe)) {
		mr_asprintf(output, "/dev/cdrom");
		log_msg(4, "Can't find cdrecord; assuming %s", output);
		mr_free(cdr_exe);
		if (!does_device_exist(output)) {
			log_msg(4, "That didn't work. Sorry.");
			mr_free(output);
			return(output);
		} else {
			return(output);
		}
	}

	mr_asprintf(command, "%s -scanbus 2> /dev/null", cdr_exe);
	fin = popen(command, "r");
	log_msg(4, "command=%s", command);
	mr_free(command);

	if (!fin) {
		log_OS_error("Cannot popen command");
		mr_free(cdr_exe);
		return(NULL);
	}

	malloc_string(phrase_one);
	phrase_one[0] = '\0';
	malloc_string(phrase_two);
	phrase_two[0] = '\0';

	malloc_string(tmp);
	tmp[0] = '\0';
	for ((void)fgets(tmp, MAX_STR_LEN, fin); !feof(fin);
		 (void)fgets(tmp, MAX_STR_LEN, fin)) {
		p = strchr(tmp, '\'');
		if (p) {
			q = strchr(++p, '\'');
			if (q) {
				for (r = q; *(r - 1) == ' '; r--);
				*r = '\0';
				strcpy(phrase_one, p);
				p = strchr(++q, '\'');
				if (p) {
					q = strchr(++p, '\'');
					if (q) {
						while (*(q - 1) == ' ') {
							q--;
						}
						*q = '\0';
						strcpy(phrase_two, p);
					}
				}
			}
		}
	}
	paranoid_pclose(fin);
	paranoid_free(phrase_one);

#ifndef __FreeBSD__
	if (strlen(phrase_two) == 0) {
		log_msg(4, "Not running phase two. String is empty.");
	} else {
		mr_asprintf(command, "dmesg | grep \"%s\" 2> /dev/null", phrase_two);
		fin = popen(command, "r");
		mr_free(command);

		if (!fin) {
			log_msg(4, "Cannot run 2nd command - non-fatal, fortunately");
		} else {
			for ((void)fgets(tmp, MAX_STR_LEN, fin); !feof(fin);
				 (void)fgets(tmp, MAX_STR_LEN, fin)) {
				log_msg(5, "--> '%s'", tmp);
				if (tmp[0] != ' ' && tmp[1] != ' ') {
					p = strchr(tmp, ':');
					if (p) {
						*p = '\0';
						if (strstr(tmp, "DVD")) {
							mr_free(dvd_last_resort);
							mr_asprintf(dvd_last_resort, "/dev/%s", tmp);
							log_msg(4, "Ignoring '%s' because it's a DVD drive", tmp);
						} else {
							mr_asprintf(output, "/dev/%s", tmp);
							found_it = TRUE;
						}
					}
				}
			}
			paranoid_pclose(fin);
		}
	}
	paranoid_free(phrase_two);
	paranoid_free(tmp);

#endif
#ifdef __FreeBSD__
	if (!found_it) {
		log_msg(4, "OK, approach 2");
		if (!(output = set_dev_to_this_if_rx_OK("/dev/cdrom"))) {
			if (!(output = set_dev_to_this_if_rx_OK("/dev/cdrom1"))) {
				if (!(output = set_dev_to_this_if_rx_OK("/dev/dvd"))) {
					if (!(output = set_dev_to_this_if_rx_OK("/dev/acd0"))) {
						if (!(output = set_dev_to_this_if_rx_OK("/dev/cd01"))) {
							if (!(output = set_dev_to_this_if_rx_OK("/dev/acd1"))) {
								if (!(output = set_dev_to_this_if_rx_OK("/dev/cd1"))) {
									mr_free(cdr_exe);
									return(NULL);
								}
							}
						}
					}
				}
			}
		}
	}
#else
	if (!found_it && strlen(dvd_last_resort) > 0) {
		log_msg(4, "Well, I'll use the DVD - %s - as a last resort", dvd_last_resort);
		mr_asprintf(output, "%s", dvd_last_resort);
		found_it = TRUE;
	}
	mr_free(dvd_last_resort);

	if (found_it) {
		mr_asprintf(tmp1, "grep \"%s=ide-scsi\" /proc/cmdline &> /dev/null", strrchr(output, '/') + 1);
		if (system(tmp1) == 0) {
			log_msg(4, "%s is not right. It's being SCSI-emulated. Continuing.", output);
			found_it = FALSE;
			mr_free(output);
		}
		mr_free(tmp1);
	}

	if (found_it) {
		log_msg(4, "(find_cdrom_device) --> '%s'", output);
		if (!does_device_exist(output)) {
			found_it = FALSE;
			log_msg(4, "OK, I was wrong, I haven't found it... yet.");
			mr_free(output);
		}
	}

	if (!found_it) {
		log_msg(4, "OK, approach 2");
		if (!(output = set_dev_to_this_if_rx_OK("/dev/scd0"))) {
			if (!(output = set_dev_to_this_if_rx_OK("/dev/sr0"))) {
				if (!(output = set_dev_to_this_if_rx_OK("/dev/cdrom"))) {
					if (!(output = set_dev_to_this_if_rx_OK("/dev/cdrom0"))) {
						if (!(output = set_dev_to_this_if_rx_OK("/dev/cdrom1"))) {
							if (!(output = set_dev_to_this_if_rx_OK("/dev/sr1"))) {
								if (!(output = set_dev_to_this_if_rx_OK("/dev/dvd"))) {
									if (!(output = set_dev_to_this_if_rx_OK(g_cdrw_drive_is_here))) {
										mr_free(cdr_exe);
										return(NULL);
									}
								}
							}
						}
					}
				}
			}
		}
	}
#endif

	if (found_it && try_to_mount) {
		if (mount_CDROM_here(output, mountpoint)) {
			log_msg(4, "[Cardigans] I've changed my mind");
			found_it = FALSE;
			mr_free(output);
		} else {
			mr_asprintf(tmp1, "%s/archives", mountpoint);
			if (!does_file_exist(tmp)) {
				log_msg(4, "[Cardigans] I'll take it back");
				found_it = FALSE;
				mr_free(output);
			} else {
				mr_asprintf(command, "umount %s", output);
				paranoid_system(command);
				mr_free(command);

				log_msg(4, "I'm confident the Mondo CD is in %s", output);
			}
			mr_free(tmp1);
		}
	}
	unlink(mountpoint);
	mr_free(mountpoint);

	if (found_it) {
		if (!does_file_exist(output)) {
			log_msg(3, "I still haven't found it.");
			mr_free(cdr_exe);
			mr_free(output);
			return (NULL);
		}
		log_msg(3, "(find_cdrom_device) --> '%s'", output);
		strcpy(the_last_place_i_found_it, output);
		strcpy(g_cdrom_drive_is_here, output);
		mr_free(cdr_exe);
		return (output);
	}

	mr_asprintf(command, "%s -scanbus | grep \"[0-9],[0-9],[0-9]\" | grep \"[D|C][V|D]\" | grep -n \"\" | grep \"%s\" | cut -d':' -f2", cdr_exe, g_cdrw_drive_is_here);
	mr_free(cdr_exe);

	log_msg(1, "command=%s", command);
	mr_asprintf(tmp1, "%s", call_program_and_get_last_line_of_output(command));
	mr_free(command);

	mr_free(output);
	if (strlen(tmp1) > 0) {
		mr_asprintf(output, "%s", tmp1);
		log_msg(4, "Finally found it at %s", output);
	} else {
		log_msg(4, "Still couldn't find it.");
	}
	mr_free(tmp1);

	return (output);
}


char *find_dvd_device()
{
	char *tmp = NULL;
	int devno = -1;
	char *output = NULL;

	if (g_dvd_drive_is_here[0]) {
		mr_asprintf(output, g_dvd_drive_is_here);
		log_msg(3, "Been there, done that. Returning %s", output);
		return (output);
	}

	mr_asprintf(tmp, "%s", call_program_and_get_last_line_of_output("dvdrecord -scanbus 2> /dev/null | grep \") '\" | grep -n \"\" | grep DVD | cut -d':' -f1"));
	log_msg(5, "tmp = '%s'", tmp);
	if (!tmp[0])
		mr_free(tmp);
		mr_asprintf(tmp, "%s", call_program_and_get_last_line_of_output ("cdrecord -scanbus 2> /dev/null | grep \") '\" | grep -n \"\" | grep DVD | cut -d':' -f1")
			);
	if (tmp[0]) {
		devno = atoi(tmp) - 1;
	}
	mr_free(tmp);

	if (devno >= 0) {
		mr_asprintf(output, "/dev/scd%d", devno);
		strcpy(g_dvd_drive_is_here, output);
		log_msg(2, "I think DVD is at %s", output);
	} else {
		log_msg(2, "I cannot find DVD");
	}

	return (output);
}





#include <sys/ioctl.h>

/**
 * Find the size of the specified @p drive, in megabytes. Uses @c ioctl calls
 * and @c dmesg.
 * @param drive The device to find the size of.
 * @return size in megabytes.
 */
long get_phys_size_of_drive(char *drive)
{
	int fd;
#if linux
	unsigned long long s = 0;
	int fileid, cylinders = 0, cylindersleft = 0;
	int cylindersize = 0;
	int gotgeo = 0;


	struct hd_geometry hdgeo;
#elif __FreeBSD__
	off_t s;
#endif

	long outvalA = -1;
	long outvalB = -1;
	long outvalC = -1;

	if ((fd = open(drive, O_RDONLY)) != -1) {
		if (ioctl(fd,
#if linux
#ifdef BLKGETSIZE64
				  BLKGETSIZE64,
#else
				  BLKGETSIZE,
#endif
#elif __FreeBSD__
				  DIOCGMEDIASIZE,
#endif
				  &s) != -1) {
			close(fd);
			// s>>11 works for older disks but not for newer ones
			outvalB =
#if linux
#ifdef BLKGETSIZE64
				s >> 20
#else
				s >> 11
#endif
#else
				s >> 20
#endif
				;
		}
	}

	if (outvalB <= 0) {
		log_msg(1, "Error getting size of %s: %s", drive, strerror(errno));
#if linux
		fileid = open(drive, O_RDONLY);
		if (fileid != -1) {
			if (ioctl(fileid, HDIO_GETGEO, &hdgeo) != -1) {
				if (hdgeo.cylinders && hdgeo.heads && hdgeo.sectors) {
					cylindersleft = cylinders = hdgeo.cylinders;
					cylindersize = hdgeo.heads * hdgeo.sectors / 2;
					outvalA = cylindersize * cylinders / 1024;
					log_msg(2, "Got Harddisk geometry, C:%d, H:%d, S:%d",
							hdgeo.cylinders, hdgeo.heads, hdgeo.sectors);
					gotgeo = 1;
				} else {
					log_msg(1, "Harddisk geometry wrong");
				}
			} else {
				log_msg(1,
						"Error in ioctl() getting new hard disk geometry (%s), resizing in unsafe mode",
						strerror(errno));
			}
			close(fileid);
		} else {
			log_msg(1, "Failed to open %s for reading: %s", drive,
					strerror(errno));
		}
		if (!gotgeo) {
			log_msg(1, "Failed to get harddisk geometry, using old mode");
		}
#endif
	}
// OLDER DISKS will give ridiculously low value for outvalB (so outvalA is returned) :)
// NEWER DISKS will give sane value for outvalB (close to outvalA, in other words) :)

	outvalC = (outvalA > outvalB) ? outvalA : outvalB;

//  log_msg (5, "drive = %s, error = %s", drive, strerror (errno));
//  fatal_error ("GPSOD: Unable to get size of drive");
	log_msg(1, "%s --> %ld or %ld --> %ld", drive, outvalA, outvalB,
			outvalC);

	return (outvalC);
}

/**
 * Determine whether @p format is supported by the kernel. Uses /proc/filesystems
 * under Linux and @c lsvfs under FreeBSD.
 * @param format The format to test.
 * @return TRUE if the format is supported, FALSE if not.
 */
bool is_this_a_valid_disk_format(char *format)
{
	char *good_formats = NULL;
	char *command = NULL;
	char *format_sz = NULL;

	FILE *pin;
	int retval;

	assert_string_is_neither_NULL_nor_zerolength(format);

	mr_asprintf(format_sz, "%s ", format);

#ifdef __FreeBSD__
	mr_asprintf(command, "lsvfs | tr -s '\t' ' ' | grep -v Filesys | grep -v -- -- | cut -d' ' -f1 | tr -s '\n' ' '");
#else
	mr_asprintf(command, "grep -v nodev /proc/filesystems | tr -s '\t' ' ' | cut -d' ' -f2 | tr -s '\n' ' '");
#endif

	pin = popen(command, "r");
	mr_free(command);

	if (!pin) {
		log_OS_error("Unable to read good formats");
		retval = 0;
	} else {
		mr_getline(&good_formats, pin);
		if (pclose(pin)) {
			log_OS_error("Cannot pclose good formats");
		}
		mr_strip_spaces(good_formats);
		mr_strcat(good_formats, " swap lvm raid ntfs-3g ntfs 7 ");	// " ntfs 7 " -- um, cheating much? :)
		if (strstr(good_formats, format_sz)) {
			retval = 1;
		} else {
			retval = 0;
		}
		mr_free(good_formats);
	}
	mr_free(format_sz);

	return (retval);
}


/** @def SWAPLIST_COMMAND The command to list the swap files/partitions in use. */

/**
 * Determine whether @p device_raw is currently mounted.
 * @param device_raw The device to check.
 * @return TRUE if it's mounted, FALSE if not.
 */
bool is_this_device_mounted(char *device_raw)
{

	/*@ pointers **************************************************** */
	FILE *fin;

	/*@ buffers ***************************************************** */
	char *incoming;
	char *device_with_tab = NULL;
	char *device_with_space = NULL;
	char *tmp = NULL;
	bool retval = FALSE;

#ifdef __FreeBSD__
#define SWAPLIST_COMMAND "swapinfo"
#else
#define SWAPLIST_COMMAND "cat /proc/swaps"
#endif

	/*@ end vars **************************************************** */

	if (device_raw == NULL) {
		return(FALSE);
	}

	malloc_string(incoming);
	if (device_raw[0] != '/' && !strstr(device_raw, ":/")) {
		log_msg(1, "%s needs to have a '/' prefixed - I'll do it",
				device_raw);
		mr_asprintf(tmp, "/%s", device_raw);
	} else {
		mr_asprintf(tmp, "%s", device_raw);
	}
	log_msg(1, "Is %s mounted?", tmp);
	if (!strcmp(tmp, "/proc") || !strcmp(tmp, "proc")) {
		log_msg(1,
				"I don't know how the heck /proc made it into the mountlist. I'll ignore it.");
		mr_free(tmp);
		return(FALSE);
	}
	mr_asprintf(device_with_tab, "%s\t", tmp);
	mr_asprintf(device_with_space, "%s ", tmp);
	mr_free(tmp);

	if (!(fin = popen("mount", "r"))) {
		log_OS_error("Cannot popen 'mount'");
		return(FALSE);
	}
	for ((void)fgets(incoming, MAX_STR_LEN - 1, fin); !feof(fin);
		 (void)fgets(incoming, MAX_STR_LEN - 1, fin)) {
		if (strstr(incoming, device_with_space)	//> incoming
			|| strstr(incoming, device_with_tab))	// > incoming)
		{
			paranoid_pclose(fin);
			paranoid_free(incoming);
			return(TRUE);
		}
	}
	mr_free(device_with_tab);
	mr_free(device_with_space);
	paranoid_pclose(fin);
	mr_asprintf(tmp, "%s | grep -E \"^%s\" > /dev/null 2> /dev/null", SWAPLIST_COMMAND, device_with_space);
	log_msg(4, "tmp (command) = '%s'", tmp);
	if (!system(tmp)) {
		retval = TRUE;
	}
	mr_free(tmp);
	paranoid_free(incoming);
	return(retval);
}

#ifdef __FreeBSD__
//                       CODE IS FREEBSD-SPECIFIC
/**
 * Create a loopback device for specified @p fname.
 * @param fname The file to associate with a device.
 * @return /dev entry for the device, or NULL if it couldn't be allocated.
 */
char *make_vn(char *fname)
{
	char *device = (char *) malloc(MAX_STR_LEN);
	char *mddevice = (char *) malloc(32);
	char command[MAX_STR_LEN];
	int vndev = 2;
	if (atoi
		(call_program_and_get_last_line_of_output
		 ("/sbin/sysctl -n kern.osreldate")) < 500000) {
		do {
			sprintf(mddevice, "vn%ic", vndev++);
			sprintf(command, "vnconfig %s %s", mddevice, fname);
			if (vndev > 10) {
				return NULL;
			}
		}
		while (system(command));
	} else {
		sprintf(command, "mdconfig -a -t vnode -f %s", fname);
		mddevice = call_program_and_get_last_line_of_output(command);
		if (!strstr(mddevice, "md")) {
			return NULL;
		}
	}
	sprintf(device, "/dev/%s", mddevice);
	return device;
}



//                       CODE IS FREEBSD-SPECIFIC
/**
 * Deallocate specified @p dname.
 * This should be called when you are done with the device created by make_vn(),
 * so the system does not run out of @c vn devices.
 * @param dname The device to deallocate.
 * @return 0 for success, nonzero for failure.
 */
int kick_vn(char *dname)
{
	char *command = NULL;
	int res = 0;

	if (strncmp(dname, "/dev/", 5) == 0) {
		dname += 5;
	}

	if (atoi
		(call_program_and_get_last_line_of_output
		 ("/sbin/sysctl -n kern.osreldate")) < 500000) {
		mr_asprintf(command, "vnconfig -d %s", dname);
	} else {
		mr_asprintf(command, "mdconfig -d -u %s", dname);
	}
	res = system(command);
	mr_free(command);
	return(res);
}
#endif


/**
 * Mount the CD-ROM at @p mountpoint.
 * @param device The device (or file if g_ISO_restore_mode) to mount.
 * @param mountpoint The place to mount it.
 * @return 0 for success, nonzero for failure.
 */
int mount_USB_here(char *device, char *mountpoint)
{
	/*@ buffer ****************************************************** */
	char *command = NULL;
	int retval;

	assert_string_is_neither_NULL_nor_zerolength(device);
	assert_string_is_neither_NULL_nor_zerolength(mountpoint);

	make_hole_for_dir(mountpoint);
	if (isdigit(device[0])) {
		return(1);
	}
	log_msg(4, "(mount_USB_here --- device=%s, mountpoint=%s", device, mountpoint);

#ifdef __FreeBSD__
	mr_asprintf(command, "mount_vfat %s %s 2>> %s", device, mountpoint, MONDO_LOGFILE);

#else
	mr_asprintf(command, "mount %s -t vfat %s 2>> %s", device, mountpoint, MONDO_LOGFILE);
#endif

	log_msg(4, command);
	retval = system(command);
	log_msg(1, "system(%s) returned %d", command, retval);
	mr_free(command);

	return (retval);
}

/**
 * Mount the CD-ROM at @p mountpoint.
 * @param device The device (or file if g_ISO_restore_mode) to mount.
 * @param mountpoint The place to mount it.
 * @return 0 for success, nonzero for failure.
 */
int mount_CDROM_here(char *device, const char *mountpoint)
{
	/*@ buffer ****************************************************** */
	char *command = NULL;
	int retval;
#ifdef __FreeBSD__
	char *dev = NULL;
#else
	char *options = NULL;
#endif

	assert_string_is_neither_NULL_nor_zerolength(mountpoint);

	make_hole_for_dir(mountpoint);

	if ((device == NULL) || (isdigit(device[0]))) {
		mr_free(device);
		device = find_cdrom_device(FALSE);
	}
#ifndef __FreeBSD__
	mr_asprintf(options, "ro");
#endif

	if (g_ISO_restore_mode) {

#ifdef __FreeBSD__
		mr_asprintf(dev, "%s", make_vn(device));
		if (!dev) {
			mr_asprintf(command, "Unable to mount ISO (make_vn(%s) failed)", device);
			fatal_error(command);
		}
		mr_free(device);
		device = dev;
#else
		mr_strcat(options, ",loop");
#endif

	}
	log_msg(4, "(mount_CDROM_here --- device=%s, mountpoint=%s", device, mountpoint);
	/*@ end vars *************************************************** */

#ifdef __FreeBSD__
	mr_asprintf(command, "mount_cd9660 -r %s %s 2>> %s", device, mountpoint, MONDO_LOGFILE);

#else
	mr_asprintf(command, "mount %s -o %s -t iso9660 %s 2>> %s", device, options, mountpoint, MONDO_LOGFILE);
	paranoid_free(options);
#endif

	log_msg(4, command);
	if (strncmp(device, "/dev/", 5) == 0) {
		retract_CD_tray_and_defeat_autorun();
	}
	retval = system(command);
	log_msg(1, "system(%s) returned %d", command, retval);
	mr_free(command);

	return (retval);
}






/**
 * Ask the user for CD number @p cd_number_i_want.
 * Sets g_current_media_number once the correct CD is inserted.
 * @param bkpinfo The backup information structure. Fields used:
 * - @c bkpinfo->backup_media_type
 * - @c bkpinfo->prefix
 * - @c bkpinfo->isodir
 * - @c bkpinfo->media_device
 * - @c bkpinfo->please_dont_eject_when_restoring
 * @param cd_number_i_want The CD number to ask for.
 */
void
insist_on_this_cd_number(int cd_number_i_want)
{

	/*@ int ************************************************************* */
	int res = 0;


	/*@ buffers ********************************************************* */
	char *tmp = NULL;
	char *mds = NULL;
	char *request = NULL;

	assert(bkpinfo != NULL);
	assert(cd_number_i_want > 0);

//  log_msg(3, "Insisting on CD number %d", cd_number_i_want);

	if (IS_THIS_A_STREAMING_BACKUP(bkpinfo->backup_media_type)) {
		log_msg(3,
				"No need to insist_on_this_cd_number when the backup type isn't CD-R(W) or NFS or ISO");
		return;
	}
	mr_asprintf(tmp, "mkdir -p " MNT_CDROM);
	run_program_and_log_output(tmp, 5);
	mr_free(tmp);

	if (g_ISO_restore_mode || bkpinfo->backup_media_type == iso || bkpinfo->backup_media_type == nfs) {
		// BERLIOS --- I'm tempted to do something about this...
		// Why unmount and remount again and again?
		log_msg(3, "Remounting CD");
		g_ISO_restore_mode = TRUE;
		if (is_this_device_mounted(MNT_CDROM)) {
			run_program_and_log_output("umount " MNT_CDROM, 5);
		}
		mr_asprintf(tmp, "mkdir -p %s/isodir &> /dev/null", bkpinfo->tmpdir);
		(void)system(tmp);
		mr_free(tmp);

		if (((bkpinfo->isodir == NULL) && (bkpinfo->nfs_remote_dir == NULL)) || (bkpinfo->prefix == NULL)) {
			fatal_error("Unable to prepare ISO file name. Please report to dev team");
		}
		if (bkpinfo->nfs_remote_dir) {
			// NFS
			mr_asprintf(tmp, "%s/%s/%s-%d.iso", bkpinfo->isodir, bkpinfo->nfs_remote_dir, bkpinfo->prefix, cd_number_i_want);
		} else {
			// ISO
			mr_asprintf(tmp, "%s/%s-%d.iso", bkpinfo->isodir, bkpinfo->prefix, cd_number_i_want);
		}
		if (!does_file_exist(tmp)) {
			mr_free(tmp);
			if (bkpinfo->nfs_remote_dir) {
				// NFS
				mr_asprintf(tmp, "%s/isodir/%s/%s-%d.iso", bkpinfo->tmpdir, bkpinfo->nfs_remote_dir, bkpinfo->prefix, cd_number_i_want);
			} else {
				// ISO
				mr_asprintf(tmp, "%s/isodir/%s-%d.iso", bkpinfo->tmpdir, bkpinfo->prefix, cd_number_i_want);
			}
			if (does_file_exist(tmp)) {
				log_msg(1, "FIXME - hacking bkpinfo->isodir from '%s' to %s/isodir", bkpinfo->isodir, bkpinfo->tmpdir);
				mr_free(bkpinfo->isodir);
				mr_asprintf(bkpinfo->isodir, "%s/isodir", bkpinfo->tmpdir);
			}
		}
		log_msg(3, "Mounting %s at %s", tmp, MNT_CDROM);
		if (mount_CDROM_here(tmp, MNT_CDROM)) {
			mr_free(tmp);
			fatal_error("Mommy!");
		}
		mr_free(tmp);
	}
	if ((res = what_number_cd_is_this()) != cd_number_i_want) {
		log_msg(3, "Currently, we hold %d but we want %d", res, cd_number_i_want);
		mds = media_descriptor_string(bkpinfo->backup_media_type);
		log_msg(3, "Insisting on %s #%d", mds, cd_number_i_want);
		mr_asprintf(request, "Please insert %s #%d and press Enter.", mds, cd_number_i_want);
		mr_free(mds);

		while (what_number_cd_is_this() != cd_number_i_want) {
			paranoid_system("sync");
			if (is_this_device_mounted(MNT_CDROM)) {
				res =
					run_program_and_log_output("umount " MNT_CDROM, FALSE);
			} else {
				res = 0;
			}
			if (res) {
				log_to_screen("WARNING - failed to unmount CD-ROM drive");
			}
			if (!bkpinfo->please_dont_eject) {
				res = eject_device(bkpinfo->media_device);
			} else {
				res = 0;
			}
			if (res) {
				log_to_screen("WARNING - failed to eject CD-ROM disk");
			}
			popup_and_OK(request);
			if (!bkpinfo->please_dont_eject) {
				inject_device(bkpinfo->media_device);
			}
			paranoid_system("sync");
		}
		mr_free(request);

		log_msg(1, "Thankyou. Proceeding...");
		g_current_media_number = cd_number_i_want;
	}
}

/* @} - end of deviceGroup */


/**
 * Ask user for details of backup/restore information.
 * Called when @c mondoarchive doesn't get any parameters.
 * @param bkpinfo The backup information structure to fill out with the user's data.
 * @param archiving_to_media TRUE if archiving, FALSE if restoring.
 * @return 0, always.
 * @bug No point of `int' return value.
 * @ingroup archiveGroup
 */
int interactively_obtain_media_parameters_from_user(bool archiving_to_media)
// archiving_to_media is TRUE if I'm being called by mondoarchive
// archiving_to_media is FALSE if I'm being called by mondorestore
{
	char *tmp = NULL;
	char *p = NULL;
	char *mds = NULL;
	char *sz_size = NULL;
	char *command = NULL;
	char *comment = NULL;
	char *prompt;
	int i;
	FILE *fin;

	malloc_string(prompt);
	malloc_string(tmp1);
	assert(bkpinfo != NULL);
	bkpinfo->nonbootable_backup = FALSE;

	// Tape, CD, NFS, ...?
	srandom(getpid());
	bkpinfo->backup_media_type =
		(g_restoring_live_from_cd) ? cdr :
		which_backup_media_type(bkpinfo->restore_data);
	if (bkpinfo->backup_media_type == none) {
		log_to_screen("User has chosen not to backup the PC");
		finish(1);
	}
	/* Why asking to remove the media with tape ?
	if (bkpinfo->backup_media_type == tape && bkpinfo->restore_data) {
		popup_and_OK("Please remove media from drive(s)");
	}
	*/
	log_msg(3, "media type = %s", bkptype_to_string(bkpinfo->backup_media_type));
	bkpinfo->cdrw_speed = (bkpinfo->backup_media_type == cdstream) ? 2 : 4;
	bkpinfo->compression_level =
		(bkpinfo->backup_media_type == cdstream) ? 1 : 5;
	bkpinfo->use_lzo =
		(bkpinfo->backup_media_type == cdstream) ? TRUE : FALSE;
	mvaddstr_and_log_it(2, 0, " ");

	// Find device's /dev (or SCSI) entry
	switch (bkpinfo->backup_media_type) {
	case cdr:
	case cdrw:
	case dvd:
	case usb:
		/* Never try to eject a USB device */
		if (bkpinfo->backup_media_type == usb) {
			bkpinfo->please_dont_eject = TRUE;
		}
		if (archiving_to_media) {
			if ((bkpinfo->backup_media_type != dvd) && (bkpinfo->backup_media_type != usb)) {
				if (ask_me_yes_or_no
					("Is your computer a laptop, or does the CD writer incorporate BurnProof technology?"))
				{
					bkpinfo->manual_cd_tray = TRUE;
				}
			}
			if ((bkpinfo->compression_level =
				 which_compression_level()) == -1) {
				log_to_screen("User has chosen not to backup the PC");
				finish(1);
			}
			mds = media_descriptor_string(bkpinfo->backup_media_type);
			mr_asprintf(comment, "What speed is your %s (re)writer?", mds);
			mr_free(bkpinfo->media_device);
			if (bkpinfo->backup_media_type == dvd) {
				bkpinfo->media_device = find_dvd_device();
				mr_asprintf(tmp, "1");
				mr_asprintf(sz_size, "%d", DEFAULT_DVD_DISK_SIZE);	// 4.7 salesman's GB = 4.482 real GB = 4482 MB
				log_msg(1, "Setting to DVD defaults");
			} else {
				mr_asprintf(bkpinfo->media_device, "%s", VANILLA_SCSI_CDROM);
				mr_asprintf(tmp, "4");
				mr_asprintf(sz_size, "%d", 650);
				log_msg(1, "Setting to CD defaults");
			}
			if ((bkpinfo->backup_media_type != dvd) && (bkpinfo->backup_media_type != usb)) {
				p = popup_and_get_string("Speed", comment, tmp);
				mr_free(tmp);

				if (p == NULL) {
					log_to_screen("User has chosen not to backup the PC");
					mr_free(comment);
					finish(1);
				}
				/* tmp now has the new value given by the user */
				tmp = p;
			}
			mr_free(comment);

			bkpinfo->cdrw_speed = atoi(tmp1);	// if DVD then this shouldn't ever be used anyway :)

			mr_asprintf(comment, "How much data (in Megabytes) will each %s store?", mds);
			mr_free(mds);
			p = popup_and_get_string("Size", comment, sz_size);
			mr_free(sz_size);
			mr_free(comment);

			if (p == NULL) {
				log_to_screen("User has chosen not to backup the PC");
				finish(1);
			}
			sz_size = p;

			for (i = 0; i <= MAX_NOOF_MEDIA; i++) {
				bkpinfo->media_size[i] = atoi(sz_size);
			}

			if (bkpinfo->media_size[0] <= 0) {
				log_to_screen("User has chosen not to backup the PC");
				finish(1);
			}
		}
		/* No break because we continue even for usb */
	case cdstream:
		mds = media_descriptor_string(bkpinfo->backup_media_type);

		mr_free(bkpinfo->media_device);
		if ((bkpinfo->disaster_recovery) && (bkpinfo->backup_media_type != usb)) {
			mr_asprintf(bkpinfo->media_device, "/dev/cdrom");
			log_msg(2, "CD-ROM device assumed to be at %s", bkpinfo->media_device);
		} else if ((bkpinfo->restore_data && (bkpinfo->backup_media_type != usb)) || bkpinfo->backup_media_type == dvd) {
			if ((bkpinfo->backup_media_type == dvd) || ((bkpinfo->media_device = find_cdrom_device(FALSE)) != NULL)) {
				mr_asprintf(comment, "Please specify your %s drive's /dev entry", mds);
				p = popup_and_get_string("Device?", comment, bkpinfo->media_device);
				mr_free(comment);

				if (p == NULL) {
					log_to_screen("User has chosen not to backup the PC");
					finish(1);
				}
				mr_free(bkpinfo->media_device);
				bkpinfo->media_device = p;
			}
			if (bkpinfo->media_device != NULL) {
				log_msg(2, "%s device found at %s", mds, bkpinfo->media_device);
			} else {
				log_msg(2, "%s device not found (bkpinfo->media_device is NULL)", mds);
			}
		} else {
			if (((bkpinfo->media_device = find_cdrw_device()) != NULL) && (bkpinfo->backup_media_type != usb)) {
				mr_free(bkpinfo->media_device);
			}
			if (bkpinfo->media_device != NULL) {
				if (bkpinfo->backup_media_type == usb) {
					mr_asprintf(tmp, "I think your %s media corresponds to %s. Is this correct?", mds, bkpinfo->media_device);
				} else {
					mr_asprintf(tmp, "I think I've found your %s burner at SCSI node %s. Is this correct? (Say no if you have an IDE burner and you are running a 2.6 kernel. You will then be prompted for further details.)", mds, bkpinfo->media_device);
				}
				if (!ask_me_yes_or_no(tmp)) {
					mr_free(bkpinfo->media_device);
				}
				mr_free(tmp);
			}
			if (!bkpinfo->media_device) {
				if (bkpinfo->backup_media_type == usb) {
					p = popup_and_get_string("/dev entry?", "What is the /dev entry of your USB Disk/Key, please?", NULL);
				} else {
					if (g_kernel_version < 2.6) {
						p = popup_and_get_string("Device node?", "What is the SCSI node of your CD (re)writer, please?", NULL);
					} else {
						p = popup_and_get_string("/dev entry?", "What is the /dev entry of your CD (re)writer, please?", NULL);
					}
				}
				if (p == NULL) {
					log_to_screen("User has chosen not to backup the PC");
					finish(1);
				}
				bkpinfo->media_device = p;
			}
		}
		mr_free(mds);

		if (bkpinfo->backup_media_type == cdstream) {
			for (i = 0; i <= MAX_NOOF_MEDIA; i++) {
				bkpinfo->media_size[i] = 650;
			}
		}
		break;
	case udev:
		if (!ask_me_yes_or_no
			("This option is for advanced users only. Are you sure?")) {
			log_to_screen("User has chosen not to backup the PC");
			finish(1);
		}
	case tape:

		mr_free(bkpinfo->media_device);
		if ((!bkpinfo->restore_mode) && ((bkpinfo->media_device = find_tape_device_and_size(NULL)) == NULL)) {
			log_msg(3, "Ok, using vanilla scsi tape.");
			mr_asprintf(bkpinfo->media_device, "%s", VANILLA_SCSI_TAPE);
			if ((fin = fopen(bkpinfo->media_device, "r"))) {
				paranoid_fclose(fin);
			} else {
				mr_free(bkpinfo->media_device);
				mr_asprintf(bkpinfo->media_device, "/dev/osst0");
			}
		}
		if (bkpinfo->media_device != NULL) {
			if ((fin = fopen(bkpinfo->media_device, "r"))) {
				paranoid_fclose(fin);
			} else {
				if (does_file_exist("/tmp/mondo-restore.cfg")) {
					mr_free(bkpinfo->media_device);
					bkpinfo->media_device = read_cfg_var("/tmp/mondo-restore.cfg", "media-dev");
				}
			}
		}
		if (bkpinfo->media_device != NULL) {
			mr_asprintf(tmp, "I think I've found your tape streamer at %s; am I right on the money?", bkpinfo->media_device);
			if (!ask_me_yes_or_no(tmp)) {
				mr_free(bkpinfo->media_device);
			}
			mr_free(tmp);
		}
		if (bkpinfo->media_device == NULL) {
			p = popup_and_get_string("Device name?", "What is the /dev entry of your tape streamer?", bkpinfo->media_device);
			if (p == NULL) {
				log_to_screen("User has chosen not to backup the PC");
				finish(1);
			}
			bkpinfo->media_device = p;
		}
		mr_asprintf(tmp, "ls -l %s", bkpinfo->media_device);
		if (run_program_and_log_output(tmp, FALSE)) {
			log_to_screen("User has not specified a valid /dev entry");
			finish(1);
		}
		mr_free(tmp);

		bkpinfo->use_obdr = ask_me_yes_or_no
			("Do you want to activate OBDR support for your tapes ?");
		bkpinfo->media_size[0] = 0;
		log_msg(4, "media_size[0] = %ld", bkpinfo->media_size[0]);
		if (bkpinfo->media_size[0] <= 0) {
			bkpinfo->media_size[0] = 0;
		}
		for (i = 1; i <= MAX_NOOF_MEDIA; i++) {
			bkpinfo->media_size[i] = bkpinfo->media_size[0];
		}
		if (archiving_to_media) {
			if ((bkpinfo->compression_level =
				 which_compression_level()) == -1) {
				log_to_screen("User has chosen not to backup the PC");
				finish(1);
			}
		}
		break;



	case nfs:
		/* Never try to eject a NFS device */
		bkpinfo->please_dont_eject = TRUE;

		/* Initiate bkpinfo nfs_mount path from running environment if not already done */
		if (bkpinfo->nfs_mount == NULL) {
			mr_asprintf(bkpinfo->nfs_mount, "%s", call_program_and_get_last_line_of_output("mount | grep \":\" | cut -d' ' -f1 | head -n1"));
		}
#ifdef __FreeBSD__
		if (TRUE)
#else
		if (!bkpinfo->disaster_recovery)
#endif
		{
			p = popup_and_get_string("NFS dir.", "Please enter path and directory where archives are stored remotely. (Mondo has taken a guess at the correct value. If it is incorrect, delete it and type the correct one.)", bkpinfo->nfs_mount);
			if (p == NULL) {
				log_to_screen("User has chosen not to backup the PC");
				finish(1);
			}
			mr_free(bkpinfo->nfs_mount);
			bkpinfo->nfs_mount = p;
			if (!bkpinfo->restore_data) {
				if ((bkpinfo->compression_level =
					 which_compression_level()) == -1) {
					log_to_screen("User has chosen not to backup the PC");
					finish(1);
				}
			}
			// check whether already mounted - we better remove
			// surrounding spaces and trailing '/' for this
			mr_strip_spaces(bkpinfo->nfs_mount);
			if (bkpinfo->nfs_mount[strlen(bkpinfo->nfs_mount) - 1] == '/')
				bkpinfo->nfs_mount[strlen(bkpinfo->nfs_mount) - 1] = '\0';
			mr_asprintf(command, "mount | grep \"%s \" | cut -d' ' -f3", bkpinfo->nfs_mount);
			mr_free(bkpinfo->isodir);
			mr_asprintf(bkpinfo->isodir, "%s", call_program_and_get_last_line_of_output(command));
			mr_free(command);

			if (!bkpinfo->restore_data) {
				mr_asprintf(comment, "How much data (in Megabytes) will each media store?");
				// BERLIOS: 4480 shouldn't be hardcoded here
				sz_size = popup_and_get_string("Size", comment, "4480");
				mr_free(comment);
				if (sz_size == NULL) {
					log_to_screen("User has chosen not to backup the PC");
					finish(1);
				}
			} else {
				mr_asprintf(sz_size, "0");
			}
			for (i = 0; i <= MAX_NOOF_MEDIA; i++) {
				bkpinfo->media_size[i] = atoi(sz_size);
			}
			mr_free(sz_size);
			if (bkpinfo->media_size[0] < 0) {
				log_to_screen("User has chosen not to backup the PC");
				finish(1);
			}
		}
		if (bkpinfo->disaster_recovery) {
			mr_asprintf(command ,"umount %s/isodir 2> /dev/null", bkpinfo->tmpdir);
			(void)system(command);
			mr_free(command);

			p = popup_and_get_string("NFS share", "Which remote NFS share should I mount?", bkpinfo->nfs_mount);
			if (p == NULL) {
				log_to_screen("User has chosen not to backup the PC");
				finish(1);
			}
			mr_free(bkpinfo->nfs_mount);
			bkpinfo->nfs_mount = p;
		}
		/* Initiate bkpinfo isodir path from running environment if mount already done */
		mr_free(bkpinfo->isodir);
		if (is_this_device_mounted(bkpinfo->nfs_mount)) {
			mr_asprintf(bkpinfo->isodir, "%s", call_program_and_get_last_line_of_output("mount | grep \":\" | cut -d' ' -f3 | head -n1"));

		} else {
			mr_asprintf(bkpinfo->isodir, "%s/nfsdir", bkpinfo->tmpdir);
			mr_asprintf(command, "mkdir -p %s", bkpinfo->isodir);
			run_program_and_log_output(command, 5);
			mr_free(command);

			mr_asprintf(tmp, "mount -t nfs -o nolock %s %s", bkpinfo->nfs_mount, bkpinfo->isodir);
			run_program_and_log_output(tmp, 3);
			mr_free(tmp);

			malloc_string(g_selfmounted_isodir);
			strcpy(g_selfmounted_isodir, bkpinfo->isodir);
		}
		if (!is_this_device_mounted(bkpinfo->nfs_mount)) {
			popup_and_OK("Please mount that partition before you try to backup to or restore from it.");
			finish(1);
		}
		p = popup_and_get_string("Directory", "Which directory within that mountpoint?", bkpinfo->nfs_remote_dir);
		if (p == NULL) {
			log_to_screen("User has chosen not to backup the PC");
			finish(1);
		}
		mr_free(bkpinfo->nfs_remote_dir);
		bkpinfo->nfs_remote_dir = p;

		// check whether writable - we better remove surrounding spaces for this
		mr_strip_spaces(bkpinfo->nfs_remote_dir);

		p = popup_and_get_string("Prefix.", "Please enter the prefix that will be prepended to your ISO filename.  Example: machine1 to obtain machine1-[1-9]*.iso files", bkpinfo->prefix);
		if (p == NULL) {
			log_to_screen("User has chosen not to backup the PC");
			finish(1);
		}
		mr_free(bkpinfo->prefix);
		bkpinfo->prefix = p;
		log_msg(3, "prefix set to %s", bkpinfo->prefix);

		log_msg(3, "Just set nfs_remote_dir to %s", bkpinfo->nfs_remote_dir);
		log_msg(3, "isodir is still %s", bkpinfo->isodir);
		break;

	case iso:
		if (!bkpinfo->disaster_recovery) {
			p = popup_and_get_string("Storage dir.", "Please enter the full path name to the directory for your ISO images.  Example: /mnt/raid0_0", bkpinfo->isodir);
			if (p == NULL) {
				log_to_screen("User has chosen not to backup the PC");
				finish(1);
			}
			bkpinfo->isodir = p;

			if (archiving_to_media) {
				if ((bkpinfo->compression_level =
					 which_compression_level()) == -1) {
					log_to_screen("User has chosen not to backup the PC");
					finish(1);
				}
				p = popup_and_get_string("ISO size.", "Please enter how big you want each ISO image to be (in megabytes). This should be less than or equal to the size of the CD-R[W]'s or DVD's you plan to backup to.", sz_size);
				if (p == NULL) {
					log_to_screen("User has chosen not to backup the PC");
					finish(1);
				}
				for (i = 0; i <= MAX_NOOF_MEDIA; i++) {
					bkpinfo->media_size[i] = atoi(p);
				}
				mr_free(p);
			} else {
				for (i = 0; i <= MAX_NOOF_MEDIA; i++) {
					bkpinfo->media_size[i] = 650;
				}
			}
		}
		p = popup_and_get_string("Prefix.", "Please enter the prefix that will be prepended to your ISO filename.  Example: machine1 to obtain machine1-[1-9]*.iso files", bkpinfo->prefix);
		if (p == NULL) {
			log_to_screen("User has chosen not to backup the PC");
			finish(1);
		}
		mr_free(bkpinfo->prefix);
		bkpinfo->prefix = p;
		log_msg(3, "prefix set to %s", bkpinfo->prefix);
		break;

	default:
		fatal_error
			("I, Mojo Jojo, shall defeat those pesky Powerpuff Girls!");
	}

	if (archiving_to_media) {

		mr_free(bkpinfo->boot_device);
#ifdef __FreeBSD__
#define EXAMPLEBD	"/dev/ad0"
		mr_asprintf(bkpinfo->boot_device, "%s", call_program_and_get_last_line_of_output("mount | grep ' / ' | head -1 | cut -d' ' -f1 | sed 's/\\([0-9]\\).*/\\1/'"));
#else
#define EXAMPLEBD	"/dev/hda"
		mr_asprintf(bkpinfo->boot_device, "%s", call_program_and_get_last_line_of_output("mount | grep ' / ' | head -1 | cut -d' ' -f1 | sed 's/[0-9].*//'"));
#endif
		i = which_boot_loader(bkpinfo->boot_device);
		if (i == 'U')			// unknown
		{

			p = popup_and_get_string("Boot device", "What is your boot device? (e.g. "EXAMPLEBD")", bkpinfo->boot_device);
#undef EXAMPLEBD
			if (p == NULL) {
				log_to_screen("User has chosen not to backup the PC");
				finish(1);
			}
			mr_free(bkpinfo->boot_device);
			bkpinfo->boot_device = p;
#ifdef __FreeBSD__
			i = which_boot_loader(bkpinfo->boot_device);
#else
			if (does_string_exist_in_boot_block(bkpinfo->boot_device, "LILO")) {
				i = 'L';
			} else
				if (does_string_exist_in_boot_block(bkpinfo->boot_device, "ELILO")) {
				i = 'E';
			} else
				if (does_string_exist_in_boot_block(bkpinfo->boot_device, "GRUB")) {
				i = 'G';
			} else {
				i = 'U';
			}
#endif
			if (i == 'U') {
				if (ask_me_yes_or_no("Unidentified boot loader. Shall I restore it byte-for-byte at restore time and hope for the best?")) {
					i = 'R';	// raw
				} else {
					log_to_screen("I cannot find your boot loader. Please run mondoarchive with parameters.");
					finish(1);
				}
			}
		}
		bkpinfo->boot_loader = i;
		mr_free(bkpinfo->include_paths);
		mr_asprintf(p, "/");
		bkpinfo->include_paths = p;

		p = popup_and_get_string("Backup paths", "Please enter paths which you want me to backup. The default is '/' (i.e. everything).", bkpinfo->include_paths);
		if (p == NULL) {
			log_to_screen("User has chosen not to backup the PC");
			finish(1);
		}
		mr_free(bkpinfo->include_paths);
		bkpinfo->include_paths = p;

		mr_asprintf(tmp, "%s", list_of_NFS_mounts_only());
		if (strlen(tmp) > 2) {
			mr_strcat(bkpinfo->exclude_paths, " %s",tmp);
		}
		mr_free(tmp);
// NTFS 
		mr_asprintf(tmp, "%s", call_program_and_get_last_line_of_output("parted2fdisk -l | grep -i ntfs | awk '{ print $1};' | tr -s '\\n' ' ' | awk '{ print $0};'"));
		if (strlen(tmp) > 2) {
			p = popup_and_get_string("NTFS partitions", "Please enter/confirm the NTFS partitions you wish to backup as well.", tmp);

			if (p == NULL) {
				log_to_screen("User has chosen not to backup the PC");
				finish(1);
			}
			strncpy(bkpinfo->image_devs, p, MAX_STR_LEN / 4);
			mr_free(p);
		}


		p = popup_and_get_string("Exclude paths", "Please enter paths which you do NOT want to backup. Separate them with spaces. NB: /tmp and /proc are always excluded. :-) Just hit 'Enter' if you want to do a full system backup.", bkpinfo->exclude_paths);
		if (p == NULL) {
			log_to_screen("User has chosen not to backup the PC");
			finish(1);
		}
		mr_free(bkpinfo->exclude_paths);
		bkpinfo->exclude_paths = p;

		p = popup_and_get_string("Temporary directory", "Please enter your temporary directory.", bkpinfo->tmpdir);
		if (p == NULL) {
			log_to_screen("User has chosen not to backup the PC");
			finish(1);
		}
		mr_free(bkpinfo->tmpdir);
		bkpinfo->tmpdir = p;

		p = popup_and_get_string("Scratch directory", "Please enter your scratch directory.", bkpinfo->scratchdir);
		if (p == NULL) {
			log_to_screen("User has chosen not to backup the PC");
			finish(1);
		}
		mr_free(bkpinfo->scratchdir);
		bkpinfo->scratchdir = p;

// Interactive mode:
#ifdef __IA64__
		bkpinfo->make_cd_use_lilo = TRUE;
#else
		bkpinfo->make_cd_use_lilo = FALSE;
#endif
		bkpinfo->backup_data = TRUE;
		bkpinfo->verify_data =
			ask_me_yes_or_no
			("Will you want to verify your backups after Mondo has created them?");

#ifndef __FreeBSD__
		if (!ask_me_yes_or_no
			("Are you confident that your kernel is a sane, sensible, standard Linux kernel? Say 'no' if you are using a Gentoo <1.4 or Debian <3.0, please."))
#endif
		{
			strcpy(bkpinfo->kernel_path, "FAILSAFE");
		}

		if (!ask_me_yes_or_no
			("Are you sure you want to proceed? Hit 'no' to abort.")) {
			log_to_screen("User has chosen not to backup the PC");
			finish(1);
		}
	} else {
		bkpinfo->restore_data = TRUE;	// probably...
	}

	if (bkpinfo->backup_media_type == iso
		|| bkpinfo->backup_media_type == nfs) {
		g_ISO_restore_mode = TRUE;
	}
#ifdef __FreeSD__
// skip
#else
	if (bkpinfo->backup_media_type == nfs) {
		log_msg(3, "I think the NFS mount is mounted at %s", bkpinfo->isodir);
	}
	log_it("isodir = %s", bkpinfo->isodir);
	if (bkpinfo->nfs_mount) {
		log_it("nfs_mount = '%s'", bkpinfo->nfs_mount);
	}
	if (bkpinfo->nfs_user) {
		log_it("nfs_user = '%s'", bkpinfo->nfs_user);
	}
#endif

	log_it("media device = %s", bkpinfo->media_device);
	log_it("media size = %ld", bkpinfo->media_size[1]);
	log_it("media type = %s", bkptype_to_string(bkpinfo->backup_media_type));
	log_it("prefix = %s", bkpinfo->prefix);
	log_it("compression = %ld", bkpinfo->compression_level);
	log_it("include_paths = '%s'", bkpinfo->include_paths);
	log_it("exclude_paths = '%s'", bkpinfo->exclude_paths);
	log_it("scratchdir = '%s'", bkpinfo->scratchdir);
	log_it("tmpdir = '%s'", bkpinfo->tmpdir);
	log_it("image_devs = '%s'", bkpinfo->image_devs);
	log_it("boot_device = '%s' (loader=%c)", bkpinfo->boot_device, bkpinfo->boot_loader);
	if (bkpinfo->media_size[0] < 0) {
		if (archiving_to_media) {
			fatal_error("Media size is less than zero.");
		} else {
			log_msg(2, "Warning - media size is less than zero.");
			bkpinfo->media_size[0] = 0;
		}
	}
	paranoid_free(sz_size);
	paranoid_free(tmp1);
	paranoid_free(prompt);
	return (0);
}




/**
 * @addtogroup utilityGroup
 * @{
 */
/**
 * Get a space-separated list of NFS devices and mounts.
 * @return The list created.
 * @note The return value points to static data that will be overwritten with each call.
 */
char *list_of_NFS_devices_and_mounts(void)
{
	char *exclude_these_devices = NULL;
	char *exclude_these_directories = NULL;
	static char result_sz[1024];

	mr_asprintf(exclude_these_directories,"%s",list_of_NFS_mounts_only());
	mr_asprintf(exclude_these_devices,"%s", call_program_and_get_last_line_of_output("tr -s '\t' ' ' < /etc/fstab | grep -E '( (coda|ncpfs|nfs|nfs4|smbfs|cifs|afs|gfs|ocfs|ocfs2|mvfs|nsspool|nsvol) )' | cut -d' ' -f1 | tr -s '\n' ' ' | awk '{print $0;}'"));
	snprintf(result_sz, 1023, "%s %s", exclude_these_directories, exclude_these_devices);
	mr_free(exclude_these_devices);
	mr_free(exclude_these_directories);
	return (result_sz);
}




/**
 * Get a space-separated list of NFS mounts.
 * @return The list created.
 * @note The return value points to static data that will be overwritten with each call.
 * @bug Even though we only want the mounts, the devices are still checked.
 */
char *list_of_NFS_mounts_only(void)
{
	char *exclude_these_directories = NULL;
	static char result_sz[512];

	mr_asprintf(exclude_these_directories,"%s", call_program_and_get_last_line_of_output("mount -t coda,ncpfs,nfs,nfs4,smbfs,cifs,afs,gfs,ocfs,ocfs2,mvfs,nsspool,nssvol | tr -s '\t' ' ' | cut -d' ' -f3 | tr -s '\n' ' ' | awk '{print $0;}'"));
	snprintf(result_sz, 511, "%s", exclude_these_directories);
	mr_free(exclude_these_directories);
	return (result_sz);
}

/* @} - end of utilityGroup */





/**
 * Create a randomly-named FIFO. The format is @p stub "." [random] [random] where
 * [random] is a random number between 1 and 32767.
 * @param store_name_here Where to store the new filename.
 * @param stub A random number will be appended to this to make the FIFO's name.
 * @ingroup deviceGroup
 */
void make_fifo(char *store_name_here, char *stub)
{
	char *tmp = NULL;

	assert_string_is_neither_NULL_nor_zerolength(stub);

	sprintf(store_name_here, "%s%d%d", stub, (int) (random() % 32768),
			(int) (random() % 32768));
	make_hole_for_file(store_name_here);
	mkfifo(store_name_here, S_IRWXU | S_IRWXG);
	mr_asprintf(tmp, "chmod 770 %s", store_name_here);
	paranoid_system(tmp);
	mr_free(tmp);
}






/**
 * Set the tmpdir and scratchdir to reside on the partition with the most free space.
 * Automatically excludes DOS, NTFS, SMB, and NFS filesystems.
 * @param bkpinfo The backup information structure. @c bkpinfo->tmpdir and @c bkpinfo->scratchdir will be set.
 * @ingroup utilityGroup
 */
void sensibly_set_scratchdir()
{
	char *tmp = NULL; 
	char *command = NULL;
	char *sz = NULL;

#ifdef __FreeBSD__
	mr_asprintf(tmp, "%s", call_program_and_get_last_line_of_output("LANGUAGE=C df -m -P -t nonfs,msdosfs,ntfs,ntfs-3g,smbfs,smb,cifs,afs,gfs,ocfs,ocfs2,mvfs,nsspool,nssvol | tr -s '\t' ' ' | grep -vE \"none|Filesystem\" | awk '{printf \"%s %s\\n\", $4, $6;}' | sort -n | tail -n1 | awk '{print $NF;}'"));
#else
	mr_asprintf(tmp, "%s", call_program_and_get_last_line_of_output("LANGUAGE=C df -m -P -x nfs -x nfs4 -x vfat -x ntfs -x ntfs-3g -x smbfs -x smb -x cifs -x afs -x gfs -x ocfs -x ocfs2 -x mvfs -x nsspool -x nssvol -x iso9660| sed 's/                  /devdev/' | tr -s '\t' ' ' | grep -vE \"none|Filesystem|/dev/shm\" | awk '{printf \"%s %s\\n\", $4, $6;}' | sort -n | tail -n1 | awk '{print $NF;}'"));
#endif

	if (tmp[0] != '/') {
		mr_asprintf(sz, "%s", tmp);
		mr_free(tmp);
		mr_asprintf(tmp, "/%s", sz);
		mr_free(sz);
	}
	if (!tmp[0]) {
		fatal_error("I couldn't figure out the scratchdir!");
	}

	/* Cleaning a potential previous scratchdir */
	if (bkpinfo->scratchdir) {
		mr_asprintf(command, "rm -Rf %s", bkpinfo->scratchdir);
		paranoid_system(command);
		mr_free(command);
	}
	mr_free(bkpinfo->scratchdir);

	mr_asprintf(bkpinfo->scratchdir , "%s/mondo.scratch.%d", tmp, (int) (random() % 32768));
	log_it("bkpinfo->scratchdir is being set to %s", bkpinfo->scratchdir);

	/* Cleaning potential previous scratchdir */
	mr_asprintf(command, "rm -Rf %s/mondo.scratch.*", tmp);
	mr_free(tmp);

	paranoid_system(command);
	mr_free(command);
}






/**
 * @addtogroup deviceGroup
 * @{
 */
/**
 * If we can read @p dev, set @return output to it.
 * If @p dev cannot be read, set @p output to NULL.
 * @param dev The device to check for.
 * @return output Set to @p dev if @p dev exists, NULL otherwise.
 */
char *set_dev_to_this_if_rx_OK(char *dev)
{
	char *command = NULL;
	char *output = NULL;

	if (!dev || dev[0] == '\0') {
		return(NULL);
	}
	log_msg(10, "Injecting %s", dev);
	inject_device(dev);
	if (!does_file_exist(dev)) {
		log_msg(10, "%s doesn't exist. Returning FALSE.", dev);
		return(NULL);
	}
	mr_asprintf(command, "dd bs=%ld count=1 if=%s of=/dev/null &> /dev/null", 512L, dev);
	if (!run_program_and_log_output(command, FALSE)
		&& !run_program_and_log_output(command, FALSE)) {
		mr_asprintf(output, "%s", dev);
		log_msg(4, "Found it - %s", dev);
	} else {
		log_msg(4, "It's not %s", dev);
	}
	mr_free(command);
	return(output);
}





/**
 * Find out what number CD is in the drive.
 * @param bkpinfo The backup information structure. The @c bkpinfo->media_device field is the only one used.
 * @return The current CD number, or -1 if it could not be found.
 * @note If the CD is not mounted, it will be mounted
 * (and remain mounted after this function returns).
 */
int what_number_cd_is_this()
{
	int cd_number = -1;
	char *mountdev = NULL;
	char *tmp = NULL;

	assert(bkpinfo != NULL);
	//  log_it("Asking what_number_cd_is_this");
	if (g_ISO_restore_mode) {
		mr_asprintf(tmp, "mount | grep iso9660 | awk '{print $3;}'");
		mr_asprintf(mountdev, "%s%s", call_program_and_get_last_line_of_output(tmp), "/archives/THIS-CD-NUMBER");
		cd_number = atoi(last_line_of_file(mountdev));
		mr_free(mountdev);
		mr_free(tmp);

		return (cd_number);
	}

	if (bkpinfo->media_device) {
		mr_asprintf(mountdev, "%s", bkpinfo->media_device);
	}
	if (!mountdev) {
		log_it("(what_number_cd_is_this) Warning - media_device unknown. Finding out...");
		mr_free(bkpinfo->media_device);
		bkpinfo->media_device = find_cdrom_device(FALSE);
		mr_asprintf(mountdev, "%s", bkpinfo->media_device);
	}
	if (!is_this_device_mounted(MNT_CDROM)) {
		if (bkpinfo->backup_media_type == usb) {
			mount_USB_here(mountdev, MNT_CDROM);
		} else {
			mount_CDROM_here(mountdev, MNT_CDROM);
		}
	}
	mr_free(mountdev);

	cd_number = atoi(last_line_of_file(MNT_CDROM "/archives/THIS-CD-NUMBER"));
	return (cd_number);
}







/**
 * Find out what device is mounted as root (/).
 * @return Root device.
 * @note The returned string points to static storage and will be overwritten with every call.
 * @bug A bit of a misnomer; it's actually finding out the root device.
 * The mountpoint (where it's mounted) will obviously be '/'.
 */
char *where_is_root_mounted()
{
	/*@ buffers **************** */
	static char tmp[MAX_STR_LEN];


#ifdef __FreeBSD__
	strcpy(tmp, call_program_and_get_last_line_of_output
		   ("mount | grep \" on / \" | cut -d' ' -f1"));
#else
	strcpy(tmp, call_program_and_get_last_line_of_output
		   ("mount | grep \" on / \" | cut -d' ' -f1 | sed s/[0-9]// | sed s/[0-9]//"));
	if (strstr(tmp, "/dev/cciss/")) {
		strcpy(tmp, call_program_and_get_last_line_of_output
			   ("mount | grep \" on / \" | cut -d' ' -f1 | cut -dp -f1"));
	}
	if (strstr(tmp, "/dev/md")) {
		strcpy(tmp,
			   call_program_and_get_last_line_of_output
			   ("mount | grep \" on / \" | cut -d' ' -f1"));
	}
#endif

	return (tmp);
}


/**
 * Find out which boot loader is in use.
 * @param which_device Device to look for the boot loader on.
 * @return 'L' for LILO, 'E'for ELILO, 'G' for GRUB, 'B' or 'D' for FreeBSD boot loaders, or 'U' for Unknown.
 * @note Under Linux, all drives are examined, not just @p which_device.
 */
#ifdef __FreeBSD__
char which_boot_loader(char *which_device)
{
	int count_lilos = 0;
	int count_grubs = 0;
	int count_boot0s = 0;
	int count_dangerouslydedicated = 0;

	log_it("looking at drive %s's MBR", which_device);
	if (does_string_exist_in_boot_block(which_device, "GRUB")) {
		count_grubs++;
	}
	if (does_string_exist_in_boot_block(which_device, "LILO")) {
		count_lilos++;
	}
	if (does_string_exist_in_boot_block(which_device, "Drive")) {
		count_boot0s++;
	}
	if (does_string_exist_in_first_N_blocks
		(which_device, "FreeBSD/i386", 17)) {
		count_dangerouslydedicated++;
	}
	log_it("%d grubs and %d lilos and %d elilos and %d boot0s and %d DD\n",
		   count_grubs, count_lilos, count_elilos, count_boot0s,
		   count_dangerouslydedicated);

	if (count_grubs && !count_lilos) {
		return ('G');
	} else if (count_lilos && !count_grubs) {
		return ('L');
	} else if (count_grubs == 1 && count_lilos == 1) {
		log_it("I'll bet you used to use LILO but switched to GRUB...");
		return ('G');
	} else if (count_boot0s == 1) {
		return ('B');
	} else if (count_dangerouslydedicated) {
		return ('D');
	} else {
		log_it("Unknown boot loader");
		return ('U');
	}
}

#else

char which_boot_loader(char *which_device)
{
	/*@ buffer ***************************************************** */
	char *list_drives_cmd = NULL;
	char *current_drive;

	/*@ pointers *************************************************** */
	FILE *pdrives;

	/*@ int ******************************************************** */
	int count_lilos = 0;
	int count_grubs = 0;

	/*@ end vars *************************************************** */

	malloc_string(current_drive);

#ifdef __IA64__
	/* No choice for it */
	return ('E');
#endif
	assert(which_device != NULL);

	mr_asprintf(list_drives_cmd, "parted2fdisk -l 2>/dev/null | grep \"/dev/.*:\" | tr -s ':' ' ' | tr -s ' ' '\n' | grep /dev/; echo %s", where_is_root_mounted());
	log_it("list_drives_cmd = %s", list_drives_cmd);

	if (!(pdrives = popen(list_drives_cmd, "r"))) {
		log_OS_error("Unable to open list of drives");
		mr_free(list_drives_cmd);
		paranoid_free(current_drive);
		return ('\0');
	}
	mr_free(list_drives_cmd);

	for ((void)fgets(current_drive, MAX_STR_LEN, pdrives); !feof(pdrives);
		 (void)fgets(current_drive, MAX_STR_LEN, pdrives)) {
		strip_spaces(current_drive);
		log_it("looking at drive %s's MBR", current_drive);
		if (does_string_exist_in_boot_block(current_drive, "GRUB")) {
			count_grubs++;
			strcpy(which_device, current_drive);
			break;
		}
		if (does_string_exist_in_boot_block(current_drive, "LILO")) {
			count_lilos++;
			strcpy(which_device, current_drive);
			break;
		}
	}
	if (pclose(pdrives)) {
		log_OS_error("Cannot pclose pdrives");
	}
	log_it("%d grubs and %d lilos\n", count_grubs, count_lilos);
	if (count_grubs && !count_lilos) {
		paranoid_free(current_drive);
		return ('G');
	} else if (count_lilos && !count_grubs) {
		paranoid_free(current_drive);
		return ('L');
	} else if (count_grubs == 1 && count_lilos == 1) {
		log_it("I'll bet you used to use LILO but switched to GRUB...");
		paranoid_free(current_drive);
		return ('G');
	} else {
		// We need to look on each partition then
		mr_asprintf(list_drives_cmd, "parted2fdisk -l 2>/dev/null | grep -E \"^/dev/\" | tr -s ':' ' ' | tr -s ' ' '\n' | grep /dev/");
		log_it("list_drives_cmd = %s", list_drives_cmd);

		if (!(pdrives = popen(list_drives_cmd, "r"))) {
			log_OS_error("Unable to open list of drives");
			mr_free(list_drives_cmd);
			paranoid_free(current_drive);
			return ('\0');
		}
		mr_free(list_drives_cmd);

		for ((void)fgets(current_drive, MAX_STR_LEN, pdrives); !feof(pdrives);
		 	(void)fgets(current_drive, MAX_STR_LEN, pdrives)) {
			strip_spaces(current_drive);
			log_it("looking at partition %s's BR", current_drive);
			if (does_string_exist_in_boot_block(current_drive, "GRUB")) {
				count_grubs++;
				strcpy(which_device, current_drive);
				break;
			}
			if (does_string_exist_in_boot_block(current_drive, "LILO")) {
				count_lilos++;
				strcpy(which_device, current_drive);
				break;
			}
		}
		if (pclose(pdrives)) {
			log_OS_error("Cannot pclose pdrives");
		}
		log_it("%d grubs and %d lilos\n", count_grubs, count_lilos);
		paranoid_free(current_drive);
		if (count_grubs && !count_lilos) {
			return ('G');
		} else if (count_lilos && !count_grubs) {
			return ('L');
		} else if (count_grubs == 1 && count_lilos == 1) {
			log_it("I'll bet you used to use LILO but switched to GRUB...");
			return ('G');
		} else {
			log_it("Unknown boot loader");
			return ('U');
		}
	}
}
#endif




/**
 * Write zeroes over the first 16K of @p device.
 * @param device The device to zero.
 * @return 0 for success, 1 for failure.
 */
int zero_out_a_device(char *device)
{
	FILE *fout;
	int i;

	assert_string_is_neither_NULL_nor_zerolength(device);

	log_it("Zeroing drive %s", device);
	if (!(fout = fopen(device, "w"))) {
		log_OS_error("Unable to open/write to device");
		return (1);
	}
	for (i = 0; i < 16384; i++) {
		fputc('\0', fout);
	}
	paranoid_fclose(fout);
	log_it("Device successfully zeroed.");
	return (0);
}

/**
 * Return the device pointed to by @p incoming.
 * @param incoming The device to resolve symlinks for.
 * @return The path to the real device file.
 * @note The returned string points to static storage that will be overwritten with each call.
 * @bug Won't work with file v4.0; needs to be written in C.
 */
char *resolve_softlinks_to_get_to_actual_device_file(char *incoming)
{
	static char output[MAX_STR_LEN];
	char *command = NULL;
	char *curr_fname;
	char *scratch = NULL;
	char *tmp = NULL;
	char *p;

	struct stat statbuf;
	malloc_string(curr_fname);
	if (!does_file_exist(incoming)) {
		log_it
			("resolve_softlinks_to_get_to_actual_device_file --- device not found");
		strcpy(output, incoming);
	} else {
		strcpy(curr_fname, incoming);
		lstat(curr_fname, &statbuf);
		while (S_ISLNK(statbuf.st_mode)) {
			log_msg(1, "curr_fname = %s", curr_fname);
			mr_asprintf(command, "file %s", curr_fname);
			mr_asprintf(tmp, "%s", call_program_and_get_last_line_of_output(command));
			mr_free(command);
			for (p = tmp + strlen(tmp); p != tmp && *p != '`' && *p != ' ';
				 p--);
			p++;
			mr_asprintf(scratch, "%s", p);
			for (p = scratch; *p != '\0' && *p != '\''; p++);
			*p = '\0';
			log_msg(0, "curr_fname %s --> '%s' --> %s", curr_fname, tmp, scratch);
			mr_free(tmp);

			if (scratch[0] == '/') {
				strcpy(curr_fname, scratch);	// copy whole thing because it's an absolute softlink
			} else {			// copy over the basename cos it's a relative softlink
				p = curr_fname + strlen(curr_fname);
				while (p != curr_fname && *p != '/') {
					p--;
				}
				if (*p == '/') {
					p++;
				}
				strcpy(p, scratch);
			}
			mr_free(scratch);
			lstat(curr_fname, &statbuf);
		}
		strcpy(output, curr_fname);
		log_it("resolved %s to %s", incoming, output);
	}
	paranoid_free(curr_fname);
	return (output);
}

/* @} - end of deviceGroup */


/**
 * Return the type of partition format (GPT or MBR)
 */
char *which_partition_format(const char *drive)
{
	static char output[4];
	char *tmp = NULL;
	char *command = NULL;
	char *fdisk = NULL;
#ifdef __IA64__
	struct stat buf;
#endif
	log_msg(0, "Looking for partition table format type");
	mr_asprintf(fdisk, "/sbin/parted2fdisk");
	log_msg(1, "Using %s", fdisk);
	mr_asprintf(command, "%s -l %s | grep 'EFI GPT'", fdisk, drive);
	mr_free(fdisk);

	mr_asprintf(tmp, "%s", call_program_and_get_last_line_of_output(command));
	mr_free(command);

	if (strstr(tmp, "GPT") == NULL) {
		strcpy(output, "MBR");
	} else {
		strcpy(output, "GPT");
	}
	mr_free(tmp);

	log_msg(0, "Found %s partition table format type", output);
	return (output);
}

/* @} - end of deviceGroup */
