/***************************************************************************
$Id: libmondo-cli.c 2202 2009-05-13 02:34:00Z bruno $
*******************************************************************/

/**
 * @file
 * Functions for handling command-line arguments passed to mondoarchive.
 */

/** @def BOOT_LOADER_CHARS The characters allowed for boot loader on this platform. */

#include <pthread.h>
#include "my-stuff.h"
#include "mondostructures.h"
#include "libmondo-cli-EXT.h"
#include "libmondo.h"
#include "mr_string.h"

extern int g_loglevel;
extern bool g_text_mode;
extern char g_startdir[MAX_STR_LEN];	///< ????? @bug ?????
extern char *MONDO_OPTIONS;

/*@ file pointer **************************************************/
extern FILE *g_tape_stream;

/*@ long long *****************************************************/
extern long long g_tape_posK;

/*@ long **********************************************************/
extern long g_noof_sets;

/*@ bool******** **************************************************/
bool g_debugging = FALSE;		///< ????? @bug ????? @ingroup globalGroup
bool g_running_live = FALSE;	///< ????? @bug ????? @ingroup globalGroup
extern bool g_cd_recovery;

extern void setup_tmpdir(char *path);
extern double g_kernel_version;
extern int g_current_media_number;
extern pid_t g_main_pid;
extern char *resolve_softlinks_to_get_to_actual_device_file(char *);

/* Stuff that handles the -I and -E option when a whole disk DSF is used */
typedef struct mounted_fs_struct {
	char device[MAX_STR_LEN];		/* The name of the device */
	char mount_point[MAX_STR_LEN];	/* The devices mount point */
	unsigned char check;			/* 1 == included on DSF */
	struct mounted_fs_struct *next;
} MOUNTED_FS_STRUCT;
static MOUNTED_FS_STRUCT *DSF_Head = NULL;		/* Points to the first entry of mounted_fs_struct list */
static MOUNTED_FS_STRUCT *DSF_Tail = NULL;		/* Points to the last entry of mounted_fs_struct list */
static void add_mounted_fs_struct (MOUNTED_FS_STRUCT *DSFptr);
static void free_mounted_fs_list (void);
static int get_dsf_mount_list (const char *dsf, char **included_dsf_list, char **excluded_dsf_list);
static int create_list_of_non_NFS_mounted_file_systems (void);
static MOUNTED_FS_STRUCT *find_mount_point_in_list (char *mount_point);
static MOUNTED_FS_STRUCT *find_device_in_list (char *device);

/* Do we use extended attributes and acl ? 
 * By default no, use --acl & --attr options to force their usage */
extern char *g_getfacl;
extern char *g_getfattr;

/* Reference to global bkpinfo */
extern struct s_bkpinfo *bkpinfo;

extern void free_MR_global_filenames(void);

/**
 * @addtogroup cliGroup
 * @{
 */
/**
 * Populate @p bkpinfo from the command-line parameters stored in @p argc and @p argv.
 * @param argc The argument count, including the program name; @p argc passed to main().
 * @param argv The argument vector; @p argv passed to main().
 * @param bkpinfo The backup information structure to populate.
 * @return The number of problems with the command line (0 for success).
 */
int
handle_incoming_parameters(int argc, char *argv[])
{
	/*@ int *** */
	int res = 0;
	int retval = 0;
	int i = 0, j;

	/*@ buffers *************** */
	char *tmp;
	char flag_val[128][MAX_STR_LEN];
	bool flag_set[128];

	tmp = malloc(9*MAX_STR_LEN);
	for (i = 0; i < 128; i++) {
		flag_val[i][0] = '\0';
		flag_set[i] = FALSE;
	}
	//  strcpy (bkpinfo->scratchdir, "/home");
	for (j = 1; j <= MAX_NOOF_MEDIA; j++) {
		bkpinfo->media_size[j] = 650;
	}							/* default */
	res =
		retrieve_switches_from_command_line(argc, argv, flag_val,
											flag_set);
	retval += res;
	if (!retval) {
		res = process_switches(flag_val, flag_set);
		retval += res;
	}
/*
  if (!retval)
    {
*/
	log_msg(3, "Switches:-");
	for (i = 0; i < 128; i++) {
		if (flag_set[i]) {
			sprintf(tmp, "-%c %s", i, flag_val[i]);
			log_msg(3, tmp);
		}
	}
//    }
	sprintf(tmp, "rm -Rf %s/tmp.mondo.*", bkpinfo->tmpdir);
	paranoid_system(tmp);
	sprintf(tmp, "rm -Rf %s/mondo.scratch.*", bkpinfo->scratchdir);
	paranoid_system(tmp);
	sprintf(bkpinfo->scratchdir + strlen(bkpinfo->scratchdir),
			"/mondo.scratch.%ld", random() % 32767);
	sprintf(tmp, "mkdir -p %s/tmpfs", bkpinfo->tmpdir);
	paranoid_system(tmp);
	sprintf(tmp, "mkdir -p %s", bkpinfo->scratchdir);
	paranoid_system(tmp);
	paranoid_free(tmp);
	return (retval);
}




/**
 * Store the sizespec(s) stored in @p value into @p bkpinfo.
 * @param bkpinfo The backup information structure; the @c bkpinfo->media_size field will be populated.
 * @param value The sizespec (e.g. "2g", "40m").
 * @return 0, always.
 * @bug Return code not needed.
 */
int process_the_s_switch(char *value)
{
	int j;
	char tmp[MAX_STR_LEN], *p, comment[MAX_STR_LEN];

	assert(bkpinfo != NULL);
	assert(value != NULL);

	bkpinfo->media_size[0] = -1;	/* dummy value */
	for (j = 1, p = value; j < MAX_NOOF_MEDIA && strchr(p, ',');
		 j++, p = strchr(p, ',') + 1) {
		strncpy(tmp, p, MAX_STR_LEN);
		*(strchr(tmp, ',')) = '\0';
		bkpinfo->media_size[j] = friendly_sizestr_to_sizelong(tmp);
		sprintf(comment, "media_size[%d] = %ld", j,
				bkpinfo->media_size[j]);
		log_msg(3, comment);
	}
	for (; j <= MAX_NOOF_MEDIA; j++) {
		bkpinfo->media_size[j] = friendly_sizestr_to_sizelong(p);
	}
//      bkpinfo->media_size[0] = bkpinfo->media_size[MAX_NOOF_MEDIA];
	for (j = 1; j <= MAX_NOOF_MEDIA; j++) {
		if (bkpinfo->media_size[j] <= 0) {
			log_msg(1, "You gave media #%d an invalid size\n", j);
			return (-1);
		}
	}
	return (0);
}

/**
 * Frees the memory for all of the structures on the linked list of 
 * all of the non-NFS mounted file systems.
 */
static void free_mounted_fs_list (void) {
	MOUNTED_FS_STRUCT *DSFptr = NULL;
	MOUNTED_FS_STRUCT *DSFnext = NULL;
 
	DSFptr = DSF_Head;
	while (DSFptr != NULL) {
		DSFnext = DSFptr->next;
		paranoid_free(DSFptr);
		DSFptr = DSFnext;
	}
	DSF_Head = NULL;
	DSF_Tail = NULL;
}

/**
 * Creates a singly linked list of all of the non-NFS mounted file systems.
 * @param DSFptr A pointer  to the structure MOUNTED_FS_STRUCT used to hold 
 * the list of mounted file systems. 
 * @return None. 
 */
static void add_mounted_fs_struct (MOUNTED_FS_STRUCT *DSFptr)
{
	assert (DSFptr);
	if (DSF_Head == NULL) {
		DSF_Head = DSFptr;
	} else {
		DSF_Tail->next = DSFptr;
	}
	DSFptr->next = NULL;
	DSF_Tail = DSFptr;
}

/**
 * Find the structure, in the singly linked list of all of the non-NFS 
 * mounted file systems, that contains the specified device.
 * @param device The device to find 
 * @return NULL if it didn't find the device, a pointer to the
 * structure if it did. 
 */
static MOUNTED_FS_STRUCT *find_device_in_list (char *device)
{
	MOUNTED_FS_STRUCT *DSFptr = NULL;

	DSFptr = DSF_Head;
	while (DSFptr != NULL) {
		if (!strcmp(DSFptr->device, device)) {
			break;
		}
		DSFptr = DSFptr->next;
	}
	return (DSFptr);
}

/**
 * Find the structure, in the singly linked list of all of the non-NFS 
 * mounted file systems, that contains the specified mount point.
 * @param mount_point The mount point to find 
 * @return NULL is it didn't find the mount point, a pointer to the
 * structure if it did. 
 */
static MOUNTED_FS_STRUCT *find_mount_point_in_list (char *mount_point)
{
	MOUNTED_FS_STRUCT *DSFptr = NULL;

	DSFptr = DSF_Head;
	while (DSFptr != NULL) {
		if (!strcmp(DSFptr->mount_point, mount_point)) {
			break;
		}
		DSFptr = DSFptr->next;
	}
	return (DSFptr);
}

/**
 * Creates a linked list of all of the non-NFS mounted file systems.
 * We use a linked list because we don't know how many  mounted file
 * there are (and there can be a lot).
 * @return 0 on success and greated than 0 on failure. 
 */
static int create_list_of_non_NFS_mounted_file_systems (void)
{
	int i = 0;
	int mount_cnt = 0;
	char *mounted_file_system = NULL;
	char *command = NULL;
	char *token = NULL;
	char token_chars[] =" :\t\r\f\a\0";
	MOUNTED_FS_STRUCT *DSFptr = NULL;

	free_mounted_fs_list();
	/********
	* Find the number of mounted file system entries and their respective mount points. 
	* I can't return all of the entries as one string because it's length can be longer
	* than MAX_STR_LEN which is used in call_program_and_get_last_line_of_output(). 
	* So start looping and get the number of  mounted file systems and query them one by one.
	********/
	/* Get the number of mounted file systems ((those that start with "/dev/" */
	asprintf(&command, "mount 2>/dev/null | awk '{if($1 ~ \"^/dev/\"){print $0}}'|wc -l");
	log_msg(5, "Running: %s", command);
	asprintf(&mounted_file_system, "%s", call_program_and_get_last_line_of_output(command));
	paranoid_free(command);

	mount_cnt = atoi(mounted_file_system);
	log_msg (5, "mount_cnt: %d", mount_cnt);
	paranoid_free(mounted_file_system);

	for (i=mount_cnt; i > 0; i--) {
		asprintf(&command, "mount 2>/dev/null | awk '{if($1 ~ \"^/dev/\"){print $1,$3}}'|head -n %d", i);
		log_msg(5, "Running: %s", command);
		asprintf(&mounted_file_system, "%s", call_program_and_get_last_line_of_output(command));
		paranoid_free(command);

		log_msg (5, "mounted_file_system: %s", mounted_file_system);
		if ((token = strtok(mounted_file_system, token_chars)) == NULL) {
			log_msg (4, "Could not get the list of mounted file systems");
			paranoid_free(mounted_file_system);
			return (1);
		}
		log_msg (5, "token: %s", token);
		while (token != NULL) {
			log_msg (5, "token: %s", token);
			if ((DSFptr = (MOUNTED_FS_STRUCT *) calloc(1, sizeof(MOUNTED_FS_STRUCT))) == NULL) {
				fatal_error ("Cannot allocate memory");
			}
			add_mounted_fs_struct(DSFptr);
			strcpy(DSFptr->device, token);
			if ((token = strtok(NULL, token_chars)) == NULL) {
				log_msg (5, "Ran out of entries on the mounted file systems list");
				paranoid_free(mounted_file_system);
				return (1);
			}
			log_msg (5, "token: %s", token);
			strcpy(DSFptr->mount_point, token);
			token = strtok(NULL, token_chars); 
		}
		paranoid_free(mounted_file_system);
	}
	/********
	* DSFptr = DSF_Head;
	* while (DSFptr != NULL) {
	* printf ("Dev: %s  MP: %s  Check: %d\n", DSFptr->device, DSFptr->mount_point, DSFptr->check);
	* DSFptr = DSFptr->next;
	* }
	********/
	return (0);
}

/**
 * Given a whole disk device special file, determine which mounted file systems
 * are on the dsf's partitions and which mounted file systems are not.
 * @param dsf The whole disk device special file.
 * @param included_dsf_list A char pointer used to hold the list of mount points
 * that are on the dsf. Memory for the array will be allocated within the function.
 * @param excluded_dsf_list A char pointer used to hold the list of mount points
 * that are not on the dsf. Memory for the array will be allocated within the function.
 * @return 0 on success, -1 if no device special file was passed in, -2 if a device
 * special file was passed in but it has no partitions on it, or 1 on failure 
 */
static int get_dsf_mount_list (const char *dsf, char **included_dsf_list, char **excluded_dsf_list) {
	int i = 0;
	int c = 0;
	int lastpos = 0;
	char VG[MAX_STR_LEN];
	char *tmp = NULL;
	char *command = NULL;
	char *partition_list = NULL;
	char partitions[64][MAX_STR_LEN];
	char *mount_list = NULL;
	char *token = NULL;
	char token_chars[] =" \t\r\f\a\0";
	MOUNTED_FS_STRUCT *DSFptr = NULL;

	memset((char *)partitions, 0, sizeof(partitions));

	log_msg(5, "dsf: %s", dsf);
	
	/********
	* See if a device special file was passed in (i.e. it must start with /dev/
	********/
	if (strncmp(dsf, "/dev/", 5)) {
		log_msg (5, "%s does not start with /dev/ and (probably) is not a  device special file", dsf);
		return (-1);
	}
	log_msg(5, "  %s looks like a device special file", dsf);
	/* Verify that the dsf exists */ 
	asprintf(&command, "ls -al %s 2>/dev/null | wc -l", dsf);
	log_msg(5, "  Executing: %s", command);
	asprintf(&tmp, "%s", call_program_and_get_last_line_of_output(command));
	paranoid_free(command);

	log_msg(5, "  Return value: %s", tmp);
	c = atoi(tmp);
	paranoid_free(tmp);

	if (!c) {
		log_to_screen("Cannot find device special file %s", dsf);
		return (1);
	}
	log_msg(5, "  %s device special file exists", dsf);
	
	/* Get a list of the mounted file systems */
	if (create_list_of_non_NFS_mounted_file_systems()) {
		log_to_screen ("Could not get the list of mounted file systems");
		return (1);
	}
	log_msg (5, "Processing dsf: %s", dsf);
	/********
	* Get a list of the dsf's partitions. There could be no partitions on the disk
	* or a dsf of a partition was passed in (e.g. /dev/sda1 instead of /dev/sda).
	* Either way, it's an error.
	********/
	asprintf(&command, 
	  "fdisk -l %s 2>/dev/null|grep -E \"^/dev/\"|awk '{printf(\"%%s \", $1)}END{print \"\"}'", dsf);
	log_msg(4, "Executing: %s", command);
	asprintf(&partition_list, "%s", call_program_and_get_last_line_of_output(command));
	paranoid_free(command);
	log_msg(4, "Partition list for %s: %s", dsf, partition_list);
	if (!strlen(partition_list)) {
		/* There were no partitions on the disk */
		log_msg(4, "Cannot find any partitions on device special file %s", dsf);
		return (-2);
	}

	/* Fill the partition list */
	i = 0;
	lastpos = 0;
	while ((token = mr_strtok(partition_list, token_chars, &lastpos)) != NULL) {
		log_msg (5, "Found partition: %s", token);
		strcpy(partitions[i++], token);
	}
	paranoid_free(partition_list);
  
	/********
	 * At this point, we have a list of all of the partitions on the dsf. Now try to
	 * see which partitions have a file system on them. 
	 *
	 * Loop through each partition on the disk and:
	 *
	 * - If the partition is swap, it ignores it.
	 *
	 * - If the partition is mounted (e.g. /dev/sda1 is mounted on /boot), it adds an entry 
	 *	to the linked list, copies to it the device name and mount point, and sets check == 1.
	 *
	 * - If the partition is part of a Volume Group that has Logical Volumes mounted, it adds 
	 *	an entry to the linked list for each mounted Logical Volume in that Volume Group, copying 
	 *	to it the device name and mount point, and sets check == 1. Note that if the Volume Group 
	 *	contains more than one disk, it will still add the entry even if the Logical Volume's 
	 *	extents are not on the dsf that was passed in to the function. For example, Volume Group 
	 *	VolGroup00 contains the disks /dev/sda1 and /dev/sdb1, and the Logical Volumes LogVol01, 
	 *	which is mounted on /var and has all of its extents on /dev/sda1, and LogVol02, which is 
	 *	mounted as /usr and has all of its extents on /dev/sdb1. If you pass /dev/sda into the 
	 *	function, both /var and /usr will be archived even though /usr is actually on/dev/sdb.
	 *
	 * - If the partition is part of a Volume Group that has Logical Volumes used in a mounted 
	 *	software raid device, it adds an entry to the linked list, copies to it the software raid 
	 *	device name and mount point, and sets check == 1.
	 *
	 * - If the partition is part of a mounted software raid device, it adds an entry to the linked 
	 *	list, copies to it the software raid device name and mount point, and sets check == 1.
	 *
	 ********/
	for (i=0; strlen(partitions[i]); i++) {
		log_msg(4, "Processing partition: %s", partitions[i]);
		/* See if it's swap. If it is, ignore it. */
		asprintf(&command,
		  "fdisk -l %s 2>/dev/null | awk '{if(($1==\"%s\")&&(toupper($0) ~ \"SWAP\")){print $1;exit}}'", 
		  dsf, partitions[i]);
		log_msg(4, "  Running: %s", command);
		asprintf(&tmp, "%s", call_program_and_get_last_line_of_output(command));
		paranoid_free(command);
		log_msg(4, "  Return value: %s", tmp);
		c = strlen(tmp);
		paranoid_free(tmp);
		if (c) {
			log_msg(4, "It's swap. Ignoring partition %s", partitions[i]); 
			continue;
		} 
		/* It's not swap. See if we can find the mount point from the mount command. */
		asprintf(&command, "mount 2>/dev/null | awk '{if((NF>0)&&($1==\"%s\")){print $3}}'", partitions[i]);
		asprintf(&tmp, "%s", call_program_and_get_last_line_of_output(command));
		paranoid_free(command);
		if (strlen(tmp)) {
			log_msg(4, "  %s is mounted: %s", partitions[i], tmp);
			if ((DSFptr = find_mount_point_in_list(tmp)) == NULL) {
				log_msg (4, "Can't find mount point %s in mounted file systems list", tmp);
				paranoid_free(tmp);
				return (1);
			}
			DSFptr->check = 1;
			paranoid_free(tmp);
			continue;
		}
		paranoid_free(tmp);
		/* It's not swap and it's not mounted. See if it's LVM */
		log_msg(4, "  It's not mounted. Checking to see if it's LVM...");
		/* Get the partition ID; 8e for LVM */
		asprintf(&command, "fdisk -l %s |awk '{if($1 ~ \"^%s\"){print $5}}'", dsf, partitions[i]);
		log_msg(4, "  Running: %s", command);
		asprintf(&tmp, "%s", call_program_and_get_last_line_of_output(command));
		paranoid_free(command);
		if (strlen(tmp)) {
			log_msg(4, "  Partition ID: %s", tmp);
			if (!strcasecmp(tmp, "8e")) { 
				/* It's LVM: Find the VG it's in */
				log_msg(4, "  It's LVM: Find the VG it's in...");
				asprintf(&command, "pvdisplay -v %s 2>/dev/null|grep \"VG Name\"|awk '{print $NF}'", partitions[i]);
				log_msg(4, "  Running: %s", command);
				strcpy(VG, call_program_and_get_last_line_of_output(command));
				paranoid_free(command);
				log_msg(4, "  Volume Group: %s", VG);
				if (strlen(VG)) {
					/* Found the Volume Group. Now find all of the VG's mount points */
					log_msg(4, "  Found the Volume Group. Now find all of the VG's mount points");
					asprintf(&command, 
					  "mount 2>/dev/null|grep -E \"/dev/mapper/%s-|/dev/%s/\"|awk '{printf(\"%%s \",$3)}END{print \"\"}'", 
					  VG, VG);
					log_msg(4, "  Running: %s", command);
					asprintf(&mount_list, "%s", call_program_and_get_last_line_of_output(command));
					paranoid_free(command);
					log_msg(4, "  VG %s mount_list: %s", VG, mount_list);
					lastpos = 0;
					while ((token = mr_strtok(mount_list, token_chars, &lastpos)) != NULL) {
						log_msg (5, "mount point token: %s", token);
						if ((DSFptr = find_mount_point_in_list(token)) == NULL) {
							log_msg (4, "Can't find mount point %s in mounted file systems list", token);
							paranoid_free(tmp);
							return (1);
						}
						DSFptr->check = 1;
					}
					/********
					 * Now we want to see if there are any software raid devices using
					 * any of the Logical Volumes on the Volume Group.
					 *******/
					paranoid_free(mount_list);
					asprintf(&command, "%s",
						"cat /proc/mdstat|grep -iv Personal|awk '{if($0~\"^.*[ ]+:\"){printf(\"/dev/%s \", $1)}}END{print \"\"}'");
					log_msg (5, "Running: %s", command);
					asprintf(&mount_list, "%s", call_program_and_get_last_line_of_output(command));
					paranoid_free(command);
					log_msg(4, "  Software raid device list: %s", mount_list);	 
					lastpos = 0;
					while ((token = mr_strtok(mount_list, token_chars, &lastpos)) != NULL) {
						asprintf(&command, "mdadm --detail %s 2>/dev/null | grep -c %s", token, VG); 
						log_msg (5, "Running: %s", command);
						paranoid_free(tmp);
						asprintf(&tmp, "%s", call_program_and_get_last_line_of_output(command));
						paranoid_free(command);
						log_msg(4, "Number of Software raid device: %s", tmp);
						if (atoi(tmp)) {
							/* This device is on our disk */
							if ((DSFptr = find_device_in_list(token)) == NULL) {
								log_msg (4, "Can't find device %s in mounted file systems list", token);
								paranoid_free(tmp);
								return (1);
							}
							DSFptr->check = 1;
						}
					}
					paranoid_free(mount_list);
				} else {
					log_msg (4, "Error finding Volume Group for partition %s", partitions[i]);
					paranoid_free(tmp);
					return (1);
				}
				paranoid_free(tmp);
				continue;
			}
		} else {
			log_msg (4, "Error finding partition type for the partition %s", partitions[i]);
		}
		paranoid_free(tmp);
		/********
		 * It's not swap, mounted, or LVM. See if it's used in a software raid device.
		 ********/
		log_msg (5, "It's not swap, mounted, or LVM. See if it's used in a software raid device.");
		asprintf(&command, "mdadm --examine %s 2>/dev/null | awk '{if($1 == \"UUID\"){print $3}}'", partitions[i]);
		log_msg(4, "  Running: %s", command);
		asprintf(&tmp, "%s", call_program_and_get_last_line_of_output(command));
		paranoid_free(command);
		if (!strlen(tmp)) {
			log_msg(4, "  Partition %s is not used in a non-LVM software raid device", partitions[i]);
			paranoid_free(tmp);
			continue;
		}
		log_msg (5, "  UUID: %s", tmp);
		/* Get the Software raid device list */
		asprintf(&command, "%s",
		 "cat /proc/mdstat|grep -iv Personal|awk '{if($0~\"^.*[ ]+:\"){printf(\"/dev/%s \", $1)}}END{print \"\"}'");
		log_msg (5, "  Running: %s", command);
		asprintf(&mount_list, "%s", call_program_and_get_last_line_of_output(command));
		paranoid_free(command);
		log_msg(4, "  Software raid device list: %s", mount_list);	 
		/* Loop through the software raid device list to see if we can find the partition */
		lastpos = 0;
		while ((token = mr_strtok(mount_list, token_chars, &lastpos)) != NULL) {
			asprintf(&command, "mdadm --detail %s 2>/dev/null | grep -c %s", token, tmp);
			log_msg(4, "  Running: %s", command);
			paranoid_free(tmp);
			asprintf(&tmp, "%s", call_program_and_get_last_line_of_output(command));
			paranoid_free(command);
			if (!atoi(tmp)) {
				log_msg (4,"  Didn't find partition %s in software raid device %s", partitions[i], token);
			} else {
				if ((DSFptr = find_device_in_list(token)) == NULL) {
					log_msg (4, "Can't find device %s in mounted file systems list", token);
					paranoid_free(tmp);
					return (1);
				}
				DSFptr->check = 1;
				break;
			}
		}
		paranoid_free(tmp);
	}
	paranoid_free(partition_list);
	paranoid_free(mount_list);

	/* Determine how much memory to allocate for included_dsf_list and excluded_dsf_list */
	i = 0;
	DSFptr= DSF_Head;
	while (DSFptr != NULL) {
		i += strlen(DSFptr->mount_point) + 1;
		DSFptr = DSFptr->next;
	}
	log_msg (5, "i: %d", i);
	if ((*included_dsf_list = (char *) calloc(i+100, sizeof(char))) == NULL) {
		fatal_error ("Cannot allocate memory");
	}
	if ((*excluded_dsf_list = (char *) calloc(i+100, sizeof(char))) == NULL) {
		fatal_error ("Cannot allocate memory");
	}
	DSFptr= DSF_Head;
	while (DSFptr != NULL) {
		if (DSFptr->check) {
			log_msg (5, "%s is mounted on %s and is on disk %s\n", DSFptr->device, DSFptr->mount_point, dsf);
			strcat(*included_dsf_list, DSFptr->mount_point);
			strcat(*included_dsf_list, " ");
		} else {
			log_msg (4, "%s is mounted on %s and is NOT on disk %s\n", DSFptr->device, DSFptr->mount_point, dsf);
			strcat(*excluded_dsf_list, DSFptr->mount_point);
			strcat(*excluded_dsf_list, " ");
		}
		DSFptr = DSFptr->next;
	}
	log_msg (5, "included_dsf_list: %s", *included_dsf_list);
	log_msg (5, "excluded_dsf_list: %s", *excluded_dsf_list);
	return (0);
}


/**
 * Process mondoarchive's command-line switches.
 * @param bkpinfo The backup information structure to populate.
 * @param flag_val An array of the argument passed to each switch (the letter is the index).
 * If a switch is not set or has no argument, the field in @p flag_val doesn't matter.
 * @param flag_set An array of <tt>bool</tt>s indexed by switch letter: TRUE if it's set,
 * FALSE if it's not.
 * @return The number of problems with the switches, or 0 for success.
 * @bug Maybe include a list of all switches (inc. intentionally undocumented ones not in the manual!) here?
 */
int
process_switches(char flag_val[128][MAX_STR_LEN], bool flag_set[128])
{

	/*@ ints *** */
	int i = 0;
	int retval = 0;
	int percent = 0;
	int lastpos = 0;

	/*@ buffers ** */
	char *tmp = NULL;
	char *tmp1 = NULL;
	char *psz = NULL;
	char *p = NULL;
	char *q = NULL;
	char *token = NULL;
	char *mounted_on_dsf = NULL;
	char *not_mounted_on_dsf = NULL;
	char token_chars[] =" \t\r\f\a\0";

	long itbs = 0L;

	struct stat buf;

	malloc_string(tmp);
	malloc_string(psz);

	assert(bkpinfo != NULL);
	assert(flag_val != NULL);
	assert(flag_set != NULL);

	bkpinfo->internal_tape_block_size = DEFAULT_INTERNAL_TAPE_BLOCK_SIZE;

	/* compulsory */
	i = flag_set['c'] + flag_set['i'] + flag_set['n'] +
		flag_set['t'] + flag_set['u'] + flag_set['r'] +
		flag_set['w'] + flag_set['C'] + flag_set['U'];
	if ((i == 0) && (! bkpinfo->restore_data)) {
		retval++;
		log_to_screen("You must specify the media type\n");
	}
	if (i > 1) {
		retval++;
		log_to_screen("Please specify only one media type\n");
	}

	if (flag_set['K']) {
		g_loglevel = atoi(flag_val['K']);
         	log_msg(1,"Loglevel forced to %d",g_loglevel);
		if (g_loglevel < 3) {
			g_loglevel = 3;
		}
	}

	if ((flag_set['L'] && flag_set['0']) && (! bkpinfo->restore_data)) {
		retval++;
		log_to_screen("You cannot have 'no compression' _and_ LZOP.\n");
	}
	if (! bkpinfo->restore_data) {
		bkpinfo->backup_data = flag_set['O'];
	}
	bkpinfo->verify_data = flag_set['V'];

	if (flag_set['I'] && !bkpinfo->backup_data) {
		log_to_screen("-I switch is ignored if just verifying");
	}
	if (flag_set['E'] && !bkpinfo->backup_data) {
		log_to_screen("-E switch is ignored if just verifying");
	}

	if (!find_home_of_exe("afio")) {
		if (find_home_of_exe("star")) {
			flag_set['R'] = TRUE;
			log_msg(1, "Using star instead of afio");
		} else {
			fatal_error
				("Neither afio nor star is installed. Please install at least one.");
		}
	}

	if (flag_set['R']) {
		bkpinfo->use_star = TRUE;
		if (flag_set['L']) {
			fatal_error("You may not use star and lzop at the same time.");
		}
		if (!find_home_of_exe("star")) {
			fatal_error
				("Please install 'star' RPM or tarball if you are going to use -R. Thanks.");
		}
	}

	if ((flag_set['W']) && (! bkpinfo->restore_data)) {
		bkpinfo->nonbootable_backup = TRUE;
		log_to_screen("Warning - you have opted for non-bootable backup");
		if (flag_set['f'] || flag_set['l']) {
			log_to_screen
				("You don't need to specify bootloader or bootdevice");
		}
	}

	if (flag_set['I']) {
		if (bkpinfo->include_paths[0] == '-') {
			retval++;
			log_to_screen("Please supply a sensible value with '-I'\n");
		}
		if (!strcmp(bkpinfo->include_paths, "/")) {
			log_msg(2, "'/' is pleonastic.");
			bkpinfo->include_paths[0] = '\0';
		}
		if (bkpinfo->include_paths[0]) {
			strcat(bkpinfo->include_paths, " ");
		}
		
		asprintf(&tmp1, "%s", flag_val['I']);
		p = tmp1;
		q = tmp1;

		/* Cut the flag_val['I'] in parts containing all paths to test them */
		while (p != NULL) {
			q = strchr(p, ' ');
			if (q != NULL) {
				*q = '\0';
				if ((stat(p, &buf) != 0) && (! bkpinfo->restore_data)) {
					log_msg(1, "ERROR ! %s doesn't exist", p);
					fatal_error("ERROR ! You specified a directory to include which doesn't exist");
				}
				p = q+1 ;
			} else {
				if ((stat(p, &buf) != 0) && (! bkpinfo->restore_data)) {
					log_msg(1, "ERROR ! %s doesn't exist", p);
					fatal_error("ERROR ! You specified a directory to include which doesn't exist");
				}
				p = NULL;
			}
		}
		paranoid_free(tmp1);
		while ((token = mr_strtok(flag_val['I'], token_chars, &lastpos)) != NULL) {
			switch (get_dsf_mount_list(token, &mounted_on_dsf, &not_mounted_on_dsf)) {
			/* It's a dsf but not a whole disk dsf */
			case -2:
				log_to_screen("Could %s be a partition instead of a whole disk device special file?\n Ignored.", token);
				break;
			/* Fatal error; exit */
			case 1: 
				fatal_error("Error processing -I option");
			/* Everything is OK; process to archive data */
			case 0:
				log_to_screen("Archiving only the following file systems on %s:\n", token);
				log_to_screen("  %s\n", mounted_on_dsf);
				strcpy(bkpinfo->include_paths, "/");
				if (strlen(not_mounted_on_dsf)) {
					log_msg (5, "Adding to bkpinfo->exclude_paths due to -I option: %s", not_mounted_on_dsf);
					log_to_screen("Not archiving the following file systems:\n");
					log_to_screen("  %s\n", not_mounted_on_dsf);
					strcat(bkpinfo->exclude_paths, not_mounted_on_dsf);
					strcat(bkpinfo->exclude_paths, "");
				}
				break;
			/* A device special file was not passed in. Process it as a path. */
			case -1:
					strcat(bkpinfo->include_paths, token);
					strcat(bkpinfo->include_paths, " ");
				break; 
			}
		}
		log_msg(1, "include_paths is now '%s'", bkpinfo->include_paths);
		if (bkpinfo->exclude_paths != NULL) {
			log_msg(1, "exclude_paths is now '%s'", bkpinfo->exclude_paths);
		}
		log_msg(4, "Finished with the -I option");
	}

	if (g_kernel_version >= 2.6 && !flag_set['d']
		&& (flag_set['c'] || flag_set['w']) && (! bkpinfo->restore_data)) {
		fatal_error
			("If you are using the 2.6.x kernel, please specify the CD-R(W) device.");
	}


	if (flag_set['J']) {
		if (flag_set['I']) {
			retval++;
			log_to_screen
				("Please do not use -J in combination with -I. If you want to make a list of files to backup, that's fine, use -J <filename> but please don't muddy the waters by combining -J with -I. Thanks. :-)");
		}
		bkpinfo->make_filelist = FALSE;
		strcpy(bkpinfo->include_paths, flag_val['J']);
	}

	if ((flag_set['c'] || flag_set['w'] || flag_set['C'] || flag_set['r']) && (! bkpinfo->restore_data)) {
		if (!flag_set['r'] && g_kernel_version <= 2.5
			&& strstr(flag_val['d'], "/dev/")) {
			fatal_error
				("Please don't give a /dev entry. Give a SCSI node for the parameter of the -d flag.");
		}
		if (flag_set['r'] && g_kernel_version <= 2.5
			&& !strstr(flag_val['d'], "/dev/")) {
			fatal_error
				("Please give a /dev entry, not a SCSI node, as the parameter of the -d flag.");
		}
		if (g_kernel_version >= 2.6 && !strstr(flag_val['d'], "/dev/")) {
			log_to_screen
				("Linus says 2.6 has a broken ide-scsi module. Proceed at your own risk...");
		}

		if (system("which cdrecord > /dev/null 2> /dev/null")
			&& system("which dvdrecord > /dev/null 2> /dev/null")) {
			fatal_error
				("Please install dvdrecord/cdrecord and try again.");
		}
		if (flag_set['C']) {
			bkpinfo->cdrw_speed = atoi(flag_val['C']);
			if (bkpinfo->cdrw_speed < 1) {
				fatal_error
					("You specified a silly speed for a CD-R[W] drive");
			}
			if (!flag_set['L']) {
				log_to_screen
					("You must use -L with -C. Therefore I am setting it for you.");
				flag_set['L'] = 1;
				flag_val['L'][0] = '\0';
			}
		} else {
			log_msg(3, "flag_val['c'] = %s", flag_val['c']);
			log_msg(3, "flag_val['w'] = %s", flag_val['w']);
//    log_msg(3, "flag_set['r'] = %i", flag_set['r'] );
			if (flag_set['c']) {
				bkpinfo->cdrw_speed = atoi(flag_val['c']);
			} else if (flag_set['w']) {
				bkpinfo->cdrw_speed = atoi(flag_val['w']);
			} else if (flag_set['r']) {
				bkpinfo->cdrw_speed = 1;	/*atoi(flag_val['r']); */
			}

			if (bkpinfo->cdrw_speed < 1) {
				fatal_error
					("You specified a silly speed for a CD-R[W] drive");
			}
		}
	}

	if ((flag_set['t'] && !flag_set['d']) && (! bkpinfo->restore_data)) {
		log_it("Hmm! No tape drive specified. Let's see what we can do.");
		if (find_tape_device_and_size(flag_val['d'], tmp)) {
			fatal_error
				("Tape device not specified. I couldn't find it either.");
		}
		flag_set['d'] = TRUE;
		sprintf(tmp,
				"You didn't specify a tape streamer device. I'm assuming %s",
				flag_val['d']);
		log_to_screen(tmp);
		percent = 0;
	}

	if (flag_set['U'])			// USB
	{
		if (! flag_set['d']) {
			fatal_error
				("You need to specify a device file with -d for bootable USB device usage");
		}
		if ((!flag_set['s']) && (! bkpinfo->restore_data)) {
			fatal_error("You did not specify a size (-s) for your USB device. Aborting");
		}
	}

	if (flag_set['r'])			// DVD
	{
		if (flag_set['m']) {
			fatal_error
				("Manual CD tray (-m) not yet supported in conjunction w/ DVD drives. Drop -m.");
		}
		if (!flag_set['d']) {
			if (!find_dvd_device(flag_val['d'], FALSE)) {
				flag_set['d'] = TRUE;
				log_to_screen("I guess DVD drive is at %s", flag_val['d']);
			}
		}
		if (strchr(flag_val['d'], ',')) {
			fatal_error
				("Please don't give a SCSI node. Give a _device_, preferably a /dev entry, for the parameter of the -d flag.");
		}
		if (! bkpinfo->restore_data) {
			if (!find_home_of_exe("growisofs")) {
				fatal_error
					("Please install growisofs (probably part of dvd+rw-tools). If you want DVD support, you need it.");
			}
			if (!find_home_of_exe("dvd+rw-format")) {
				fatal_error
					("Please install dvd+rw-format (probably part of dvd+rw-tools). If you want DVD support, you need it.");
			}
			if (!flag_set['s']) {
				sprintf(flag_val['s'], "%d", DEFAULT_DVD_DISK_SIZE);	// 4.7 salesman's GB = 4.482 real GB = 4582 MB
				strcat(flag_val['s'], "m");
				log_to_screen
					("You did not specify a size (-s) for DVD. I'm guessing %s.",
				 	flag_val['s']);
				flag_set['s'] = 1;
			}
		}
/*
      if (flag_set['Z']) {
	  bkpinfo->blank_dvd_first = TRUE;
      }
*/
	}

	if (flag_set['t'] || flag_set['u']) {	/* tape size */
		if (strchr(flag_val['d'], ',')) {
			fatal_error
				("Please don't give a SCSI node. Give a _device_, preferably a /dev entry, for the parameter of the -d flag.");
		}
		if ((flag_set['O']) && (! bkpinfo->restore_data)) {
			if (flag_set['s']) {
				if (flag_set['t']) {
					fatal_error
						("For the moment, please don't specify a tape size. Mondo should handle end-of-tape gracefully anyway.");
				}
				if (process_the_s_switch(flag_val['s'])) {
					fatal_error("Bad -s switch");
				}
			} else if (flag_set['u'] || flag_set['t']) {
				for (i = 0; i <= MAX_NOOF_MEDIA; i++) {
					bkpinfo->media_size[i] = 0;
				}
			} else {
				retval++;
				log_to_screen("Tape size not specified.\n");
			}
		}
	} else if (! bkpinfo->restore_data) {			/* CD|USB size */
		if (flag_set['s']) {
			if (process_the_s_switch(flag_val['s'])) {
				fatal_error("Bad -s switch");
			}
		}
		if (flag_set['w']) {
			bkpinfo->wipe_media_first = TRUE;
		}						/* CD-RW */
	}

	if (flag_set['n']) {
		strncpy(bkpinfo->nfs_mount, flag_val['n'], MAX_STR_LEN);
		if (!flag_set['d']) {
			strncpy(bkpinfo->nfs_remote_dir, "/", MAX_STR_LEN);
		}
		sprintf(tmp, "mount | grep -E \"^%s .*\" | cut -d' ' -f3",
				bkpinfo->nfs_mount);
		strncpy(bkpinfo->isodir,
				call_program_and_get_last_line_of_output(tmp),
				MAX_STR_LEN / 4);
		if (strlen(bkpinfo->isodir) < 3) {
			retval++;
			log_to_screen("NFS share is not mounted. Please mount it.\n");
		}
		log_msg(3, "mount = %s", bkpinfo->nfs_mount);
		log_msg(3, "isodir= %s", bkpinfo->isodir);
	}

	if (flag_set['c']) {
		bkpinfo->backup_media_type = cdr;
	}
	if (flag_set['C']) {
		bkpinfo->backup_media_type = cdstream;
	}
	if (flag_set['i']) {
		bkpinfo->backup_media_type = iso;
	}
	if (flag_set['n']) {
		bkpinfo->backup_media_type = nfs;
		/* Never try to eject a NFS device */
		bkpinfo->please_dont_eject = TRUE;
	}
	if (flag_set['r']) {
		bkpinfo->backup_media_type = dvd;
	}
	if (flag_set['t']) {
		bkpinfo->backup_media_type = tape;
	}
	if (flag_set['u']) {
		bkpinfo->backup_media_type = udev;
	}
	if (flag_set['w']) {
		bkpinfo->backup_media_type = cdrw;
	}
	if (flag_set['U']) {
		bkpinfo->backup_media_type = usb;
		/* Never try to eject a USB device */
		bkpinfo->please_dont_eject = TRUE;
	}
	if (flag_set['z']) {
		if (find_home_of_exe("getfattr")) {
			asprintf(&g_getfattr,"getfattr");
		}
		if (find_home_of_exe("getfacl")) {
			asprintf(&g_getfacl,"getfacl");
		}
	}

	/* optional, popular */
	if (flag_set['g']) {
		g_text_mode = FALSE;
	}

	if (flag_set['E']) {
		if (bkpinfo->exclude_paths[0] == '-') {
			retval++;
			log_to_screen("Please supply a sensible value with '-E'\n");
		}
		if (bkpinfo->exclude_paths[0]) {
			strcat(bkpinfo->exclude_paths, " ");
		}
		asprintf(&tmp1, "%s", flag_val['E']);

		p = tmp1;
		q = tmp1;

		/* Cut the flag_val['E'] in parts containing all paths to test them */
		while (p != NULL) {
			q = strchr(p, ' ');
			if (q != NULL) {
				*q = '\0';
				/* Fix bug 14 where ending / cause a problem later 
				* so handled here for the moment */
				q--;
				if (*q == '/') {
					*q = '\0';
				}
				q++;
				/* End of bug fix */
				if ((stat(p, &buf) != 0) && (! bkpinfo->restore_data)) {
					log_msg(1, "WARNING ! %s doesn't exist", p);
				}
				p = q+1 ;
			} else {
				if ((stat(p, &buf) != 0) && (! bkpinfo->restore_data)) {
					log_msg(1, "WARNING ! %s doesn't exist", p);
				}
				p = NULL;
			}
		}
		paranoid_free(tmp1);
		lastpos = 0;
		while ((token = mr_strtok(flag_val['E'], token_chars, &lastpos)) != NULL) {
			switch (get_dsf_mount_list(token, &mounted_on_dsf, &not_mounted_on_dsf)) {
			case 1: 
				log_msg(1, "WARNING ! a path doesn't exist in -E option");
				break;
			/* Everything is OK; proceed to archive data */
			case 0:
				if (strlen(mounted_on_dsf)) {
					log_to_screen("Excluding the following file systems on %s:\n", token);
					log_to_screen("  %s\n", mounted_on_dsf);
					log_msg (5, "Adding to bkpinfo->exclude_paths due to -E option: %s", mounted_on_dsf);
					strcat(bkpinfo->exclude_paths, mounted_on_dsf);
					strcat(bkpinfo->exclude_paths, " ");
				}
				break;
			/* It's a dsf but not a whole disk dsf */
			case -2:
				log_to_screen("Could %s be a partition instead of a whole disk device special file?\nIgnored.", token);
				break;
			/* A device special file was not passed in. Process it as a path. */
			case -1:
				strcat(bkpinfo->exclude_paths, token);
				strcat(bkpinfo->exclude_paths, " ");
				break;
			}
		}
		log_msg(1, "exclude_paths is now '%s'", bkpinfo->exclude_paths);
		log_msg(4, "Finished with the -E option");
	}

	if (flag_set['e']) {
		bkpinfo->please_dont_eject = TRUE;
	}

	if ((flag_set['N'])	&& (! bkpinfo->restore_data))		// exclude NFS mounts & devices
	{
//      strncpy(psz, list_of_NFS_devices_and_mounts(), MAX_STR_LEN);
		strncpy(psz, list_of_NFS_mounts_only(), MAX_STR_LEN);
		if (bkpinfo->exclude_paths[0]) {
			strncat(bkpinfo->exclude_paths, " ", 4*MAX_STR_LEN);
		}
		strncat(bkpinfo->exclude_paths, psz, 4*MAX_STR_LEN);
		log_msg(3, "-N means we're now excluding %s",
				bkpinfo->exclude_paths);
	}

	if (strlen(bkpinfo->exclude_paths) >= 4*MAX_STR_LEN) {
		fatal_error
			("Your '-E' parameter is too long. Increase MAX_STR_LEN");
	}

	if (flag_set['b']) {
		strcpy(psz, flag_val['b']);
		log_msg(1, "psz = '%s'", psz);
		if (psz[strlen(psz) - 1] == 'k') {
			psz[strlen(psz) - 1] = '\0';
			itbs = atol(psz) * 1024L;
		} else {
			itbs = atol(psz);
		}
		log_msg(1, "'%s' --> %ld", flag_val['b'], itbs);
		log_msg(1, "Internal tape block size is now %ld bytes", itbs);
		if (itbs % 512 != 0 || itbs < 256 || itbs > 1024L * 1024) {
			fatal_error
				("Are you nuts? Silly, your internal tape block size is. Abort, I shall.");
		}
		bkpinfo->internal_tape_block_size = itbs;
	}

	if ((flag_set['D']) && (! bkpinfo->restore_data)) {
		bkpinfo->differential = 1;
//      bkpinfo->differential = atoi (flag_val['D']);
		if ((bkpinfo->differential < 1) || (bkpinfo->differential > 9)) {
			fatal_error
				("The D option should be between 1 and 9 inclusive");
		}
	}

	if (flag_set['x']) {
		strncpy(bkpinfo->image_devs, flag_val['x'], MAX_STR_LEN / 4);
		if ((run_program_and_log_output("which ntfsclone", 2)) && (! bkpinfo->restore_data)) {
			fatal_error("Please install ntfsprogs package/tarball.");
		}
	}

	if (flag_set['m']) {
		bkpinfo->manual_cd_tray = TRUE;
	}

	if ((flag_set['k']) && (! bkpinfo->restore_data)) {
		strncpy(bkpinfo->kernel_path, flag_val['k'], MAX_STR_LEN);
		if (!strcmp(bkpinfo->kernel_path, "failsafe")) {
			strcpy(bkpinfo->kernel_path, "FAILSAFE");
		}
		if (strcmp(bkpinfo->kernel_path, "FAILSAFE")
			&& !does_file_exist(bkpinfo->kernel_path)) {
			retval++;
			sprintf(tmp,
					"You specified kernel '%s', which does not exist\n",
					bkpinfo->kernel_path);
			log_to_screen(tmp);
		}
	}

	if (flag_set['p']) {
		strncpy(bkpinfo->prefix, flag_val['p'], MAX_STR_LEN / 4);
		log_msg(1,"Prefix forced to %s",bkpinfo->prefix);
	}

	if (flag_set['d']) {		/* backup directory (if ISO/NFS) */
		if (flag_set['i']) {
			strncpy(bkpinfo->isodir, flag_val['d'], MAX_STR_LEN / 4);
			sprintf(tmp, "ls -l %s", bkpinfo->isodir);
			if (run_program_and_log_output(tmp, 2)) {
				fatal_error
					("output folder does not exist - please create it");
			}
		} else if (flag_set['n']) {
			strncpy(bkpinfo->nfs_remote_dir, flag_val['d'], MAX_STR_LEN);
		} else {				/* backup device (if tape/CD-R/CD-RW) */
			strncpy(bkpinfo->media_device, flag_val['d'], MAX_STR_LEN / 4);
		}
	}

	if ((flag_set['n']) && (! bkpinfo->restore_data)) {
		asprintf(&tmp1,"%s/%s/.dummy.txt", bkpinfo->isodir,bkpinfo->nfs_remote_dir);
		sprintf(tmp, "echo hi > %s", tmp1);
		if (run_program_and_log_output(tmp, 2)) {
			retval++;
			sprintf(tmp,
					"Are you sure directory '%s' exists in remote dir '%s'?\nIf so, do you have rights to write to it?\n",
					bkpinfo->nfs_remote_dir, bkpinfo->nfs_mount);
			log_to_screen(tmp);
		}
		unlink(tmp1);
		paranoid_free(tmp1);
	}

	if (!flag_set['d']
		&& (flag_set['c'] || flag_set['w'] || flag_set['C'])) {
		if (g_kernel_version >= 2.6) {
			if (popup_and_get_string
				("Device", "Please specify the device",
				 bkpinfo->media_device, MAX_STR_LEN / 4)) {
				retval++;
				log_to_screen("User opted to cancel.");
			}
		} else if (find_cdrw_device(bkpinfo->media_device)) {
			retval++;
			log_to_screen
				("Tried and failed to find CD-R[W] drive automatically.\n");
		} else {
			flag_set['d'] = TRUE;
			strncpy(flag_val['d'], bkpinfo->media_device, MAX_STR_LEN / 4);
		}
	}

	if ((!flag_set['d'] && !flag_set['n'] && !flag_set['C']) && (! bkpinfo->restore_data)) {
		retval++;
		log_to_screen("Please specify the backup device/directory.\n");
		fatal_error
			("You didn't use -d to specify the backup device/directory.");
	}

	for (i = '0'; i <= '9'; i++) {
		if (flag_set[i]) {
			bkpinfo->compression_level = i - '0';
		}						/* not '\0' but '0' */
	}

	if (flag_set['S']) {
		sprintf(bkpinfo->scratchdir, "%s/mondo.scratch.%ld", flag_val['S'],
				random() % 32768);
	}

	if (flag_set['T']) {
		setup_tmpdir(flag_val['T']);
		sprintf(tmp, "touch %s/.foo.dat", bkpinfo->tmpdir);
		if (run_program_and_log_output(tmp, 1)) {
			retval++;
			log_to_screen
				("Please specify a tempdir which I can write to. :)");
			fatal_error("I cannot write to the tempdir you specified.");
		}
		sprintf(tmp, "ln -sf %s/.foo.dat %s/.bar.dat", bkpinfo->tmpdir, bkpinfo->tmpdir);
		if (run_program_and_log_output(tmp, 1)) {
			retval++;
			log_to_screen
				("Please don't specify a SAMBA or VFAT or NFS tmpdir.");
			fatal_error("I cannot write to the tempdir you specified.");
		}
	}

	if ((flag_set['A']) && (! bkpinfo->restore_data)) {
		strncpy(bkpinfo->call_after_iso, flag_val['A'], MAX_STR_LEN);
	}

	if ((flag_set['B']) && (! bkpinfo->restore_data)) {
		strncpy(bkpinfo->call_before_iso, flag_val['B'], MAX_STR_LEN);
	}

	if ((flag_set['H']) && (! bkpinfo->restore_data)) {
		g_cd_recovery = TRUE;
	}

	if ((flag_set['l']) && (! bkpinfo->restore_data)) {
#ifdef __FreeBSD__
#  define BOOT_LOADER_CHARS "GLBMR"
#else
#  ifdef __IA64__
#    define BOOT_LOADER_CHARS "GER"
#  else
#    define BOOT_LOADER_CHARS "GLR"
#  endif
#endif
		if (!strchr
			(BOOT_LOADER_CHARS,
			 (bkpinfo->boot_loader = flag_val['l'][0]))) {
			log_msg(1, "%c? WTF is %c? I need G, L, E or R.",
					bkpinfo->boot_loader, bkpinfo->boot_loader);
			fatal_error
				("Please specify GRUB, LILO, ELILO  or RAW with the -l switch");
		}
#undef BOOT_LOADER_CHARS
	}

	if (flag_set['f']) {
		strncpy(bkpinfo->boot_device,
				resolve_softlinks_to_get_to_actual_device_file(flag_val
															   ['f']),
				MAX_STR_LEN / 4);
	}

	if ((flag_set['P']) && (! bkpinfo->restore_data)) {
		strncpy(bkpinfo->postnuke_tarball, flag_val['P'], MAX_STR_LEN);
	}

	if (flag_set['Q']) {
		i = which_boot_loader(tmp);
		log_msg(3, "boot loader is %c, residing at %s", i, tmp);
		printf("boot loader is %c, residing at %s\n", i, tmp);
		finish(0);
	}

	if ((flag_set['L']) && (! bkpinfo->restore_data)) {
		bkpinfo->use_lzo = TRUE;
		if (run_program_and_log_output("which lzop", 2)) {
			retval++;
			log_to_screen
				("Please install LZOP. You can't use '-L' until you do.\n");
		}
	}

	if ((flag_set['G']) && (! bkpinfo->restore_data)) {
		bkpinfo->use_gzip = TRUE;
		if (run_program_and_log_output("which gzip", 2)) {
			retval++;
			log_to_screen
				("Please install gzip. You can't use '-G' until you do.\n");
		}
	}

	bkpinfo->use_obdr = FALSE;
	if (flag_set['o']) {
		if ((!flag_set['t']) && (! bkpinfo->restore_data)) {
			log_to_screen("OBDR support is only available for tapes. Use the -t option");
			fatal_error("Aborting");
		}
		bkpinfo->use_obdr = TRUE;
	}

#ifndef __FreeBSD__
	if ((!is_this_a_valid_disk_format("vfat")) && (! bkpinfo->restore_data)) {
		bkpinfo->make_cd_use_lilo = TRUE;
		log_to_screen
			("Your kernel appears not to support vfat filesystems. I am therefore");
		log_to_screen
			("using LILO instead of SYSLINUX as the media boot loader.");
	}
	if ((run_program_and_log_output("which mkfs.vfat", 2)) && (! bkpinfo->restore_data)) {
		bkpinfo->make_cd_use_lilo = TRUE;
#ifdef __IA32__
		log_to_screen
			("Your filesystem is missing 'mkfs.vfat', so I cannot use SYSLINUX as");
		log_to_screen
			("your boot loader. I shall therefore use LILO instead.");
#endif
#ifdef __IA64__
		log_to_screen
			("Your filesystem is missing 'mkfs.vfat', so I cannot prepare the EFI");
		log_to_screen("environment correctly. Please install it.");
		fatal_error("Aborting");
#endif
	}
#ifdef __IA64__
	/* We force ELILO usage on IA64 */
	bkpinfo->make_cd_use_lilo = TRUE;
#endif
#endif

	if (! bkpinfo->restore_data) {
		i = flag_set['O'] + flag_set['V'];
		if (i == 0) {
			retval++;
			log_to_screen("Specify backup (-O), verify (-V) or both (-OV).\n");
		}
	}

	if ((! bkpinfo->restore_data) && (flag_set['Z'])) {
			fatal_error
				("The -Z switch is only valid in restore mode");
	}

	if (flag_set['Z']) {
		if (! strcmp(flag_val['Z'], "nuke")) {
			bkpinfo->restore_mode = nuke;
		} else if (! strcmp(flag_val['Z'], "interactive")) {
			bkpinfo->restore_mode = interactive;
		} else if (! strcmp(flag_val['Z'], "compare")) {
			bkpinfo->restore_mode = compare;
		} else if (! strcmp(flag_val['Z'], "mbr")) {
			bkpinfo->restore_mode = mbr;
		} else if (! strcmp(flag_val['Z'], "iso")) {
			bkpinfo->restore_mode = isoonly;
		} else if (! strcmp(flag_val['Z'], "isonuke")) {
			bkpinfo->restore_mode = isonuke;
		} else {
			bkpinfo->restore_mode = interactive;
		}
	}

/* and finally... */

	paranoid_free(tmp);
	paranoid_free(psz);
	return (retval);
}



/**
 * Get the switches from @p argc and @p argv using getopt() and place them in
 * @p flag_set and @p flag_val.
 * @param argc The argument count (@p argc passed to main()).
 * @param argv The argument vector (@p argv passed to main()).
 * @param flag_val An array indexed by switch letter - if a switch is set and
 * has an argument then set flag_val[switch] to that argument.
 * @param flag_set An array indexed by switch letter - if a switch is set then
 * set flag_set[switch] to TRUE, else set it to FALSE.
 * @return The number of problems with the command line (0 for success).
 */
int
retrieve_switches_from_command_line(int argc, char *argv[],
									char flag_val[128][MAX_STR_LEN],
									bool flag_set[128])
{
	/*@ ints ** */
	int opt = 0;
	char *tmp = NULL;
	int i = 0;
	int len;

	/*@ bools *** */
	bool bad_switches = FALSE;

	assert(flag_val != NULL);
	assert(flag_set != NULL);

	for (i = 0; i < 128; i++) {
		flag_val[i][0] = '\0';
		flag_set[i] = FALSE;
	}
	while ((opt =
			getopt(argc, argv, MONDO_OPTIONS))
		   != -1) {
		if (opt == '?') {
			bad_switches = TRUE;
			/*log_it("Invalid option: %c\n",optopt); */
		} else {
			if (flag_set[optopt]) {
				bad_switches = TRUE;
				(void) asprintf(&tmp, "Switch -%c previously defined as %s\n", opt,
						flag_val[i]);
				log_to_screen(tmp);
				paranoid_free(tmp);
			} else {
				flag_set[opt] = TRUE;
				if (optarg) {
					len = strlen(optarg);
					if (optarg[0] != '/' && optarg[len - 1] == '/') {
						optarg[--len] = '\0';
						log_to_screen
							("Warning - param '%s' should not have trailing slash!",
							 optarg);
					}
					if (opt == 'd') {
						if (strchr(flag_val[opt], '/')
							&& flag_val[opt][0] != '/') {
							(void)asprintf(&tmp,
									"-%c flag --- must be absolute path --- '%s' isn't absolute",
									opt, flag_val[opt]);
							log_to_screen(tmp);
							paranoid_free(tmp);
							bad_switches = TRUE;
						}
					}
					strcpy(flag_val[opt], optarg);
				}
			}
		}
	}
	for (i = optind; i < argc; i++) {
		bad_switches = TRUE;
		(void)asprintf(&tmp, "Invalid arg -- %s\n", argv[i]);
		log_to_screen(tmp);
		paranoid_free(tmp);
	}
	return (bad_switches);
}




/**
 * Print a not-so-helpful help message and exit.
 */
void help_screen()
{
	log_msg(1, "Type 'man mondoarchive' for more information\n");
	exit(1);
}


/**
 * Terminate Mondo in response to a signal.
 * @param sig The signal number received.
 */
void terminate_daemon(int sig)
{
	char tmp[64];
	char tmp2[MAX_STR_LEN];
	//  char command[512];
	//  pid_t pid;

	switch (sig) {
	case SIGINT:
		sprintf(tmp, "SIGINT");
		strcpy(tmp2, "You interrupted me :-)");
		break;
	case SIGKILL:
		sprintf(tmp, "SIGKILL");
		strcpy(tmp2,
			   "I seriously have no clue how this signal even got to me. Something's wrong with your system.");
		break;
	case SIGTERM:
		sprintf(tmp, "SIGTERM");
		strcpy(tmp2, "Got terminate signal");
		break;
	case SIGHUP:
		sprintf(tmp, "SIGHUP");
		strcpy(tmp2, "Hangup on line");
		break;
	case SIGSEGV:
		sprintf(tmp, "SIGSEGV");
		strcpy(tmp2,
			   "Internal programming error. Please send a backtrace as well as your log.");
		break;
	case SIGPIPE:
		sprintf(tmp, "SIGPIPE");
		strcpy(tmp2, "Pipe was broken");
		break;
	case SIGABRT:
		sprintf(tmp, "SIGABRT");
		sprintf(tmp2,
				"Abort - probably failed assertion. I'm sleeping for a few seconds so you can read the message.");
		break;
	default:
		sprintf(tmp, "(Unknown)");
	}

	strcat(tmp, " signal received from OS");
	log_to_screen(tmp);
	log_to_screen(tmp2);
	if (sig == SIGABRT) {
		sleep(10);
	}
	kill_buffer();

	free_MR_global_filenames();

	fatal_error
		("MondoRescue is terminating in response to a signal from the OS");
	finish(254);				// just in case
}




/**
 * Turn signal-trapping on or off.
 * @param on If TRUE, turn it on; if FALSE, turn it off (we still trap it, just don't do as much).
 */
void set_signals(int on)
{
	int signals[] =
		{ SIGTERM, SIGHUP, SIGTRAP, SIGABRT, SIGINT, SIGKILL, SIGSTOP, 0 };
	int i;

	signal(SIGPIPE, sigpipe_occurred);
	for (i = 0; signals[i]; i++) {
		if (on) {
			signal(signals[i], terminate_daemon);
		} else {
			signal(signals[i], termination_in_progress);
		}
	}
}




/**
 * Exit immediately without cleaning up.
 * @param sig The signal we are exiting due to.
 */
void termination_in_progress(int sig)
{
	log_msg(1, "Termination in progress");
	usleep(1000);
	pthread_exit(0);
}

/* @} - end of cliGroup */
