/* libmondo-fifo.c
   $Id: libmondo-fifo.c 1663 2007-09-27 10:21:18Z bruno $
*/

/**
 * @file
 * Functions to handle buffering of tape archives as they are read/written.
 * This used the external program @c buffer mostly.
 */

#include <unistd.h>
#include <stdio.h>
#ifndef S_SPLINT_S
#include <signal.h>
#endif
#include <fcntl.h>

#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/sem.h>
#include <sys/wait.h>
#ifndef S_SPLINT_S
#include <pthread.h>
#endif

#include "my-stuff.h"
#include "mondostructures.h"
#include "libmondo.h"
#include "mr_mem.h"
#include "mr_msg.h"
#include "mr_gettext.h"

/**
 * @addtogroup globalGroup
 * @{
 */
/**
 * The SIGPIPE handler sets this to TRUE.
 */
bool g_sigpipe = FALSE;

/**
 * PID of the "main" process.
 */
pid_t g_mastermind_pid = 0;

/**
 * Command line with which @c buffer was invoked.
 */
char g_sz_call_to_buffer[MAX_STR_LEN];

/**
 * Size of the buffer used with @c buffer.
 */
int g_tape_buffer_size_MB = 0;

/* @} - end of globalGroup */

extern char *ps_options;
extern char *ps_proc_id;
extern char *MONDO_LOGFILE;

/**
 * @addtogroup fifoGroup
 * @{
 */
/**
 * Open a pipe to/from @c buffer.
 * If buffer does not work at all, we use `dd'.
 * @param device The device to read from/write to.
 * @param direction @c 'r' (reading) or @c 'w' (writing).
 * @return A file pointer to/from the @c buffer process.
 */
FILE *open_device_via_buffer(char *device, char direction,
							 long internal_tape_block_size)
{
	char *sz_dir;
	char keych;
	char *tmp;
	FILE *fres;
	int bufsize;				// in megabytes
	int res;
	int wise_upper_limit;
	int wise_lower_limit;

	assert_string_is_neither_NULL_nor_zerolength(device);
	assert(direction == 'w' || direction == 'r');
	mr_asprintf(&sz_dir, "%c", direction);
	wise_upper_limit = (am_I_in_disaster_recovery_mode()? 8 : 32);
	wise_lower_limit = 1;		// wise_upper_limit/2 + 1;
	sync();
	for (bufsize = wise_upper_limit, res = -1;
		 res != 0 && bufsize >= wise_lower_limit; bufsize--) {
		mr_asprintf(&tmp,
				 "dd if=/dev/zero bs=1024 count=16k 2> /dev/null | buffer -o /dev/null -s %ld -m %d%c",
				 internal_tape_block_size, bufsize, 'm');
		res = run_program_and_log_output(tmp, 2);
		mr_free(tmp);
	}
	if (!res) {
		bufsize++;
		mr_asprintf(&tmp, _("Negotiated max buffer of %d MB "), bufsize);
		log_to_screen(tmp);
		mr_free(tmp);
	} else {
		bufsize = 0;
		res = 0;
		log_to_screen
			(_("Cannot negotiate a buffer of ANY size. Using dd instead."));
	}
	if (direction == 'r') {
		keych = 'i';
	} else {
		keych = 'o';
	}
	if (bufsize) {
		sprintf(g_sz_call_to_buffer,
				"buffer -m %d%c -p%d -B -s%ld -%c %s 2>> %s", bufsize, 'm',
				(direction == 'r') ? 20 : 75, internal_tape_block_size,
				keych, device, MONDO_LOGFILE);
	} else {
		sprintf(g_sz_call_to_buffer, "dd bs=%ld %cf=%s",
				internal_tape_block_size, keych, device);
	}
	mr_msg(2, "Calling buffer --- command = '%s'", g_sz_call_to_buffer);
	fres = popen(g_sz_call_to_buffer, sz_dir);
	mr_free(sz_dir);
	if (fres) {
		mr_msg(2, "Successfully opened ('%c') tape device %s", direction,
				device);
	} else {
		mr_msg(2, "Failed to open ('%c') tape device %s", direction,
				device);
	}
	sleep(2);
	mr_asprintf(&tmp, "ps %s | grep \"%s\"", ps_options, g_sz_call_to_buffer);
	if (run_program_and_log_output(tmp, 2)) {
		mr_msg(2, "Warning - I think I failed to open tape, actually.");
	}
	mr_free(tmp);
	g_tape_buffer_size_MB = bufsize;
	mr_asprintf(&tmp, "ps %s | grep buffer | grep -v grep", ps_options);
	if (run_program_and_log_output(tmp, 1)) {
		fres = NULL;
		log_to_screen(_("Failed to open tape streamer. Buffer error."));
	} else {
		log_to_screen(_("Buffer successfully started."));
	}
	mr_free(tmp);
	return (fres);
}


/**
 * Kill @c buffer processes.
 * Only called in mondoarchive
 */
void kill_buffer()
{
	char *tmp = NULL;
	char *command = NULL;

	sync();
	if (g_sz_call_to_buffer == NULL) {
		return;
	}
	if (strcmp(g_sz_call_to_buffer,"") == 0) {
		return;
	}
	mr_asprintf(&command,
			"ps %s | grep -F \"%s\" | grep -Fv grep | awk '{print $2;}' | grep -v PID | head -1", ps_options, g_sz_call_to_buffer);
	mr_msg(2, "kill_buffer() --- command = %s", command);
	mr_asprintf(&tmp, call_program_and_get_last_line_of_output(command));
	mr_free(command);

	mr_asprintf(&command, "kill %s", tmp);
	mr_msg(2, "kill_buffer() --- command = %s", command);
	if (strlen(tmp) > 0) {
		run_program_and_log_output(command, TRUE);
	}
	mr_free(tmp);
	mr_free(command);
}


/**
 * Handler for SIGPIPE.
 * @param sig The signal that occurred (hopefully SIGPIPE).
 */
void sigpipe_occurred(int sig)
{
	g_sigpipe = TRUE;
}

/* @} - end of fifoGroup */

/* BERLIOS: useless ?
int 
extract_config_file_from_ramdisk(
				  char *ramdisk_fname, 
				  char *output_cfg_file, 
				  char *output_mountlist_file);
*/
