/***************************************************************************
* $Id: mondoarchive.c 1256 2007-03-21 11:48:32Z bruno $
*/


/**
 * @file
 * The main file for mondoarchive.
 */

/************************* #include statements *************************/
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "my-stuff.h"
#include "mondostructures.h"
#include "libmondo.h"
#include "mondo-cli-EXT.h"

#include "mondoarchive.h"
#include "mr_mem.h"
#include "mr_str.h"
#include "mr_msg.h"
#include "mr_file.h"
#include "mr_conf.h"

// for CVS
//static char cvsid[] = "$Id: mondoarchive.c 1256 2007-03-21 11:48:32Z bruno $";

/************************* external variables *************************/
extern void set_signals(int);
extern int g_current_media_number;
extern void register_pid(pid_t, char *);
extern int g_currentY;
extern bool g_text_mode;
extern char *g_boot_mountpt;
extern bool g_remount_cdrom_at_end, g_remount_floppy_at_end;
extern char *g_tmpfs_mountpt;
extern char *g_erase_tmpdir_and_scratchdir;
extern char *g_cdrw_drive_is_here;
extern double g_kernel_version;
extern char *g_magicdev_command;
extern t_bkptype g_backup_media_type;
extern int g_loglevel;

static char *g_cdrom_drive_is_here = NULL;
static char *g_dvd_drive_is_here = NULL;

struct mr_ar_conf mr_conf;

/***************** global vars ******************/
bool g_skip_floppies;
long diffs;
char *ps_options = "auxww";
char *ps_proc_id = "$2";

/****************** subroutines used only here ******************/


/**
 * Print a "don't panic" message to the log and a message about the logfile to the screen.
 */
static void welcome_to_mondoarchive(void)
{
	mr_msg(0, "Mondo Archive v%s --- http://www.mondorescue.org", PACKAGE_VERSION);
	mr_msg(0, "running on %s architecture", get_architecture());
	mr_msg(0, "-----------------------------------------------------------");
	mr_msg(0, "NB: Mondo logs almost everything, so don't panic if you see");
	mr_msg(0, "some error messages.  Please read them carefully before you");
	mr_msg(0, "decide to break out in a cold sweat.    Despite (or perhaps");
	mr_msg(0, "because of) the wealth of messages. some users are inclined");
	mr_msg(0, "to stop reading this log. If Mondo stopped for some reason,");
	mr_msg(0, "chances are it's detailed here.  More than likely there's a");
	mr_msg(0, "message at the very end of this log that will tell you what");
	mr_msg(0, "is wrong. Please read it!                          -Devteam");
	mr_msg(0, "-----------------------------------------------------------");

	mr_msg(0, "Zero...");
	mr_msg(1, "One...");
	mr_msg(2, "Two...");
	mr_msg(3, "Three...");
	mr_msg(4, "Four...");
	mr_msg(5, "Five...");
	mr_msg(6, "Six...");
	mr_msg(7, "Seven...");
	mr_msg(8, "Eight...");
	printf("See %s for details of backup run.\n", MONDO_LOGFILE);
}

/**
 * Do whatever is necessary to insure a successful backup on the Linux distribution
 * of the day.
 */
static void distro_specific_kludges_at_start_of_mondoarchive(void)
{
	mr_msg(2, "Unmounting old ramdisks if necessary");
	stop_magicdev_if_necessary();	// for RH+Gnome users
	run_program_and_log_output
		("umount `mount | grep shm | grep mondo | cut -d' ' -f3`", 2);
	unmount_supermounts_if_necessary();	// for Mandrake users whose CD-ROMs are supermounted
	mount_boot_if_necessary();	// for Gentoo users with non-mounted /boot partitions
	clean_up_KDE_desktop_if_necessary();	// delete various misc ~/.* files that get in the way
}


/**
 * Undo whatever was done by distro_specific_kludges_at_start_of_mondoarchive().
 */
static void distro_specific_kludges_at_end_of_mondoarchive(void)
{
	mr_msg(2, "Restarting magicdev if necessary");
	sync();
	restart_magicdev_if_necessary();	// for RH+Gnome users

	mr_msg(2, "Restarting supermounts if necessary");
	sync();
	remount_supermounts_if_necessary();	// for Mandrake users

	mr_msg(2, "Unmounting /boot if necessary");
	sync();
	unmount_boot_if_necessary();	// for Gentoo users
}

/* create the mr_ar_conf structure from mondo's conf file */
static void mr_ar_store_conf(struct mr_ar_conf *mr_conf) {
	
	mr_asprintf(mr_conf->iso_creation_cmd, mr_conf_sread("mondo_iso_creation_cmd"));
	mr_asprintf(mr_conf->iso_creation_options, mr_conf_sread("mondo_iso_creation_options"));
	mr_asprintf(mr_conf->iso_burning_cmd, mr_conf_sread("mondo_iso_burning_cmd"));
	mr_asprintf(mr_conf->iso_burning_options, mr_conf_sread("mondo_iso_burning_options"));
	iso_burning_speed = mr_conf_iread("mondo_iso_burning_speed");
	media_size = mr_conf_iread("mondo_media_size");
	mr_asprintf(mr_conf->media_device, mr_conf_sread("mondo_media_device"));
	manual_tray = mr_conf_bread("mondo_manual_tray");
	log_level = mr_conf_iread("mondo_log_level");
	mr_asprintf(mr_conf->prefix, mr_conf_sread("mondo_prefix"));
	external_tape_blocksize = mr_conf_iread("mondo_external_tape_blocksize");
	internal_tape_blocksize = mr_conf_iread("mondo_internal_tape_blocksize");
	slice_size = mr_conf_iread("mondo_slice_size");
	mr_asprintf(mr_conf->deplist_file, mr_conf_sread("mondo_deplist_file"));
	write_boot_floppy = mr_conf_bread("mondo_write_boot_floppy");
	create_mindi_cd = mr_conf_bread("mondo_create_mindi_cd");
	mr_asprintf(mr_conf->kernel, mr_conf_sread("mondo_kernel"));
	mr_asprintf(mr_conf->additional_modules, mr_conf_sread("mondo_additional_modules"));
	mr_asprintf(mr_conf->boot_loader, mr_conf_sread("mondo_boot_loader"));
	differential = mr_conf_bread("mondo_differential");
	mr_asprintf(mr_conf->compression_tool, mr_conf_sread("mondo_compression_tool"));
	compression_level = mr_conf_iread("mondo_compression_level");
	mr_asprintf(mr_conf->exclude_paths, mr_conf_sread("mondo_exclude_paths"));
	mr_asprintf(mr_conf->include_paths, mr_conf_sread("mondo_include_paths"));
	mr_asprintf(mr_conf->ui_mode, mr_conf_sread("mondo_ui_mode"));
	automatic_restore = mr_conf_bread("mondo_automatic_restore");
	mr_asprintf(mr_conf->scratch_dir, mr_conf_sread("mondo_scratch_dir"));
	mr_asprintf(mr_conf->tmp_dir, mr_conf_sread("mondo_tmp_dir"));
	mr_asprintf(mr_conf->images_dir, mr_conf_sread("mondo_images_dir"));
}

/*-----------------------------------------------------------*/



/**
 * Backup/verify the user's data.
 * What did you think it did, anyway? :-)
 */
int main(int argc, char *argv[])
{
	struct s_bkpinfo *bkpinfo = NULL;
	struct stat stbuf;
	char *tmp = NULL;
	int res = 0;
	int retval = 0;
	char *say_at_end = NULL;
	char *say_at_end2 = NULL;

#ifdef ENABLE_NLS
	setlocale(LC_ALL, "");
	(void) textdomain("mondo");
#endif
	printf(_("Initializing..."));

	/* initialize log file with time stamp */
	/* We start with a loglevel of 4 - Adapted later on */
	/* It's mandatory to set this up first as all mr_ functions rely on it */
	unlink(MONDO_LOGFILE);
	mr_msg_init(MONDO_LOGFILE,4);
	mr_msg(0, _("Time started: %s"), mr_date());

	/* Make sure I'm root; abort if not */
	if (getuid() != 0) {
		mr_log_exit(127, _("Please run as root."));
	}

	/* If -V, -v or --version then echo version no. and quit */
	if (argc == 2
		&& (!strcmp(argv[argc - 1], "-v") || !strcmp(argv[argc - 1], "-V")
			|| !strcmp(argv[argc - 1], "--version"))) {
		printf(_("mondoarchive v%s\nSee man page for help\n"), PACKAGE_VERSION);
		mr_exit(0, NULL);
	}

	/* Conf file management */
	/* Check md5 sum before */
	/* Get content */
	if (mr_conf_open(MONDO_CONF_DIR"/mondo.conf.dist") != 0) {
			mr_log_exit(-1, "Unable to open "MONDO_CONF_DIR"/mondo.conf.dist");
	}
	mr_ar_store_conf(&mr_conf);
	mr_conf_close();

	/* Add MONDO_SHARE + other environment variables for mindi */
	setenv_mondo_var();

	/* Initialize variables */
	malloc_libmondo_global_strings();
	diffs = 0;
	bkpinfo = mr_malloc(sizeof(struct s_bkpinfo));
	if (stat(MONDO_CACHE, &stbuf) != 0) {
		mr_mkdir(MONDO_CACHE,0x755);
	}

	/* BERLIOS: Hardcoded to be improved */
	unlink(MONDO_CACHE"/mindi.conf");
	unlink(MONDORESTORECFG);

	/* Configure the bkpinfo structure, global file paths, etc. */
	g_main_pid = getpid();
	mr_msg(9, "This");

	register_pid(g_main_pid, "mondo");
	set_signals(TRUE);			// catch SIGTERM, etc.
	run_program_and_log_output("dmesg -n1", TRUE);

	mr_msg(9, "Next");
	welcome_to_mondoarchive();
	distro_specific_kludges_at_start_of_mondoarchive();
	g_kernel_version = get_kernel_version();

	if (argc == 4 && !strcmp(argv[1], "getfattr")) {
		g_loglevel = 10;
		g_text_mode = TRUE;
		setup_newt_stuff();
		if (!strstr(argv[2], "filelist")) {
			mr_msg(1,_("Sorry - filelist goes first\n"));
			finish(1);
		} else {
			finish(get_fattr_list(argv[2], argv[3]));
		}
		finish(0);
	}
	if (argc == 4 && !strcmp(argv[1], "setfattr")) {
		g_loglevel = 10;
		g_text_mode = TRUE;
		setup_newt_stuff();
		finish(set_fattr_list(argv[2], argv[3]));
	}

	if (argc == 3 && !strcmp(argv[1], "wildcards")) {
		g_loglevel = 10;
		g_text_mode = TRUE;
		setup_newt_stuff();
		tmp = mr_stresc(argv[2], WILDCHARS, BACKSLASH);
		printf("in=%s; out=%s\n", argv[2], tmp);
		mr_free(tmp);
		finish(1);
	}

	if (argc == 4 && !strcmp(argv[1], "getfacl")) {
		g_loglevel = 10;
		g_text_mode = TRUE;
		setup_newt_stuff();
		if (!strstr(argv[2], "filelist")) {
			mr_msg(1,_("Sorry - filelist goes first\n"));
			finish(1);
		} else {
			finish(get_acl_list(argv[2], argv[3]));
		}
		finish(0);
	}
	if (argc == 4 && !strcmp(argv[1], "setfacl")) {
		g_loglevel = 10;
		g_text_mode = TRUE;
		setup_newt_stuff();
		finish(set_acl_list(argv[2], argv[3]));
	}

	if (argc > 2 && !strcmp(argv[1], "find-cd")) {
		g_loglevel = 10;
		g_text_mode = TRUE;
		setup_newt_stuff();
		malloc_string(tmp);
		if (find_cdrw_device(tmp)) {
			mr_msg(1,_("Failed to find CDR-RW drive\n"));
		} else {
			mr_msg(1,_("CD-RW is at %s\n"), tmp);
		}
		tmp[0] = '\0';
		if (find_cdrom_device(tmp, atoi(argv[2]))) {
			mr_msg(1,_("Failed to find CD-ROM drive\n"));
		} else {
			mr_msg(1,_("CD-ROM is at %s\n"), tmp);
		}
		mr_free(tmp);
		finish(0);
	}

	if (argc > 2 && !strcmp(argv[1], "find-dvd")) {
		g_loglevel = 10;
		g_text_mode = TRUE;
		setup_newt_stuff();
		malloc_string(tmp);
		if (find_dvd_device(tmp, atoi(argv[2]))) {
			mr_msg(1,_("Failed to find DVD drive\n"));
		} else {
			mr_msg(1,_("DVD is at %s\n"), tmp);
		}
		mr_free(tmp);
		finish(0);
	}

	if (argc > 2 && !strcmp(argv[1], "disksize")) {
		printf("%s --> %ld\n", argv[2], get_phys_size_of_drive(argv[2]));
		finish(0);
	}
	if (argc > 2 && !strcmp(argv[1], "test-dev")) {
		if (is_dev_an_NTFS_dev(argv[2])) {
			mr_msg(1,_("%s is indeed an NTFS dev\n"), argv[2]);
		} else {
			mr_msg(1,_("%s is _not_ an NTFS dev\n"), argv[2]);
		}
		finish(0);
	}

	if (pre_param_configuration(bkpinfo)) {
		fatal_error
			("Pre-param initialization phase failed. Please review the error messages above, make the specified changes, then try again. Exiting...");
	}
	sprintf(g_erase_tmpdir_and_scratchdir, "rm -Rf %s %s", bkpinfo->tmpdir,
			bkpinfo->scratchdir);

	/* Process command line, if there is one. If not, ask user for info. */
	if (argc == 1) {
		g_text_mode = FALSE;
		setup_newt_stuff();
		res = interactively_obtain_media_parameters_from_user(bkpinfo, TRUE);	/* yes, archiving */
		if (res) {
			fatal_error
				("Syntax error. Please review the parameters you have supplied and try again.");
		}
	} else {
		res = handle_incoming_parameters(argc, argv, bkpinfo);
		if (res) {
			mr_msg(1,
				_("Errors were detected in the command line you supplied.\n"));
			mr_msg(1,_("Please review the log file - %s \n"),MONDO_LOGFILE);
			mr_msg(1, "Mondoarchive will now exit.");
			finish(1);
		}
		setup_newt_stuff();
	}

/* Finish configuring global structures */
	if (post_param_configuration(bkpinfo)) {
		fatal_error
			("Post-param initialization phase failed. Perhaps bad parameters were supplied to mondoarchive? Please review the documentation, error messages and logs. Exiting...");
	}

	log_to_screen
		(_("BusyBox's sources are available from http://www.busybox.net"));

	sprintf(g_erase_tmpdir_and_scratchdir, "rm -Rf %s %s", bkpinfo->tmpdir,
			bkpinfo->scratchdir);

	/* If we're meant to backup then backup */
	if (bkpinfo->backup_data) {
		res = backup_data(bkpinfo);
		retval += res;
		if (res) {
			mr_strcat(say_at_end,
				   _("Data archived. Please check the logs, just as a precaution. "));
		} else {
			mr_strcat(say_at_end, _("Data archived OK. "));
		}
	}

	/* If we're meant to verify then verify */
	if (bkpinfo->verify_data) {
		res = verify_data(bkpinfo);
		if (res < 0) {
			mr_asprintf(&say_at_end2, _("%d difference%c found."), -res,
					(-res != 1) ? 's' : ' ');
			res = 0;
		}
		retval += res;
	}

	/* Offer to write floppy disk images to physical disks */
	if (bkpinfo->backup_data && !g_skip_floppies) {
		res = offer_to_write_boot_floppies_to_physical_disks(bkpinfo);
		retval += res;
	}

	/* Report result of entire operation (success? errors?) */
	if (!retval) {
		mvaddstr_and_log_it(g_currentY++, 0,
							_("Backup and/or verify ran to completion. Everything appears to be fine."));
	} else {
		mvaddstr_and_log_it(g_currentY++, 0,
							_("Backup and/or verify ran to completion. However, errors did occur."));
	}

	if (does_file_exist("/var/cache/mindi/mondorescue.iso")) {
		log_to_screen
			(_("/var/cache/mindi/mondorescue.iso, a boot/utility CD, is available if you want it."));
	}


	if (length_of_file("/tmp/changed.files") > 2) {
		if (g_text_mode) {
			log_to_screen
				(_("Type 'less /tmp/changed.files' to see which files don't match the archives"));
		} else {
			mr_msg(1,
					_("Type 'less /tmp/changed.files' to see which files don't match the archives"));
			mr_msg(2, "Calling popup_changelist_from_file()");
			popup_changelist_from_file("/tmp/changed.files");
			mr_msg(2, "Returned from popup_changelist_from_file()");
		}
	} else {
		unlink("/tmp/changed.files");
	}
	log_to_screen(say_at_end);
	mr_free(say_at_end);
	if (say_at_end2 != NULL) {
		log_to_screen(say_at_end2);
		mr_free(say_at_end2);
	}

	mr_asprintf(&tmp, "umount %s/tmpfs", bkpinfo->tmpdir);
	run_program_and_log_output(tmp, TRUE);
	mr_free(tmp);
	run_program_and_log_output(g_erase_tmpdir_and_scratchdir, TRUE);

	run_program_and_log_output("mount", 2);

	system("rm -f /var/cache/mondo-archive/last-backup.aborted");
	system("rm -Rf /tmp.mondo.* /mondo.scratch.*");
	if (!retval) {
		printf(_("Mondoarchive ran OK.\n"));
	} else {
		printf(_("Errors occurred during backup. Please check logfile.\n"));
	}
	distro_specific_kludges_at_end_of_mondoarchive();
	register_pid(0, "mondo");
	set_signals(FALSE);
	chdir("/tmp");				// just in case there's something wrong with g_erase_tmpdir_and_scratchdir
	system(g_erase_tmpdir_and_scratchdir);
	free_libmondo_global_strings();
	mr_free(bkpinfo);

	unlink("/tmp/filelist.full");
	unlink("/tmp/filelist.full.gz");

	if (!g_cdrom_drive_is_here) {
		mr_msg(10, "FYI, g_cdrom_drive_is_here was never used");
	}
	if (!g_dvd_drive_is_here) {
		mr_msg(10, "FYI, g_dvd_drive_is_here was never used");
	}

	/* finalize log file with time stamp */
	mr_msg(1, "Time finished: %s", mr_date());
	mr_msg_close();

	if (!g_text_mode) {
		popup_and_OK
			(_("Mondo Archive has finished its run. Please press ENTER to return to the shell prompt."));
		log_to_screen(_("See %s for details of backup run."), MONDO_LOGFILE);
		finish(retval);
	} else {
		printf(_("See %s for details of backup run.\n"), MONDO_LOGFILE);
		exit(retval);
	}

	return EXIT_SUCCESS;
}
